#' @title Predictions of excess hazard and net Survival from a \code{bsplines}
#'  object
#'
#' @description Function to predict excess hazard and net survival based on
#'  an object of class \code{bsplines}. The function allows the
#'  predictions at several time points but not exceeding the maximum time of
#'  follow-up from the baseline model.
#'
#'
#' @param object an object of class \code{bsplines}
#'
#' @param new.data new.data where is covariates
#'
#' @param times.pts time in year scale to calculate the excess hazard. The
#' default value is NULL. In this case, time variable must be provided in the
#' new.data
#'
#' @param baseline default is survival baseline; put \code{baseline = FALSE}
#'  to estimate the net survival with covariates
#'
#' @param ... additional arguments affecting the predictions of excess hazard
#' and net survival
#'
#' @keywords predict.bsplines
#'
#' @return An object of class predxhaz, which is a list of data.frame. Each
#' element of the list contains the estimates of hazard and survival at a fixed
#'  time point. The return of this function can be used to produce graphics of
#'  excess hazard or net survival, when times.pts argument is provided. This
#'  object contains:
#'
#' \item{times.pts}{the times value in year at which the excess hazard
#'   and or the net survival have been estimated}
#'
#' \item{hazard}{the excess hazard values based on the model of interest}
#'
#' \item{survival}{the net survival values based on the model of interest}
#'
#'
#'
#' @author Juste Goungounga, Robert Darlin Mba, Nathalie Graff\'eo and Roch Giorgi
#'
#' @references Goungounga JA, Touraine C, Graff\'eo N, Giorgi R;
#' CENSUR working survival group. Correcting for misclassification
#' and selection effects in estimating net survival in clinical trials.
#' BMC Med Res Methodol. 2019 May 16;19(1):104.
#' doi: 10.1186/s12874-019-0747-3. PMID: 31096911; PMCID: PMC6524224.
#' (\href{https://pubmed.ncbi.nlm.nih.gov/31096911/}{PubMed})
#'
#' Touraine C, Graff\'eo N, Giorgi R; CENSUR working survival group.
#' More accurate cancer-related excess mortality through correcting
#' background mortality for extra variables.
#' Stat Methods Med Res. 2020 Jan;29(1):122-136.
#' doi: 10.1177/0962280218823234. Epub 2019 Jan 23. PMID: 30674229.
#' (\href{https://pubmed.ncbi.nlm.nih.gov/30674229/}{PubMed})
#'
#' Mba RD, Goungounga JA, Graff\'eo N, Giorgi R; CENSUR working survival group.
#' Correcting inaccurate background mortality in excess hazard models
#' through breakpoints. BMC Med Res Methodol. 2020 Oct 29;20(1):268.
#' doi: 10.1186/s12874-020-01139-z. PMID: 33121436; PMCID: PMC7596976.
#'   (\href{https://pubmed.ncbi.nlm.nih.gov/33121436/}{PubMed})
#'
#'
#'
#'
#' @seealso \code{\link{xhaz}}, \code{\link{print.bsplines}}, \code{\link{print.constant}}
#'
#' @examples
#'
#' \donttest{
#' library("survival")
#' library("numDeriv")
#' library("survexp.fr")
#' library("splines")
#' data("dataCancer", package = "xhaz")   # load the data set in the package
#'
#' fit.phBS <- xhaz(
#'         formula = Surv(obs_time_year, event) ~ ageCentre + immuno_trt,
#'         data = dataCancer, ratetable = survexp.fr,
#'         interval = c(0, NA, NA, max(dataCancer$obs_time_year)),
#'         rmap = list(age = 'age', sex = 'sexx', year = 'year_date'),
#'         baseline = "bsplines", pophaz = "classic")
#'
#'
#' print(fit.phBS)
#'
#'
#' predicted <- predict(object = fit.phBS,
#'                      new.data = dataCancer[1:10,],
#'                      times.pts = c(seq(0,10,1)),
#'                      baseline = TRUE)
#'
#'
#' #a list of predicted hazard and survival at different time points
#' print(predicted)
#'
#'
#' #predicted hazard and survival at time points 10 years
#' do.call("cbind",predicted)[11,]
#' }
#' @export

predict.bsplines <- function(object,
                           new.data = NULL,
                           times.pts = NULL,
                           baseline = TRUE,
                           ...) {
    Call <- match.call()
    int  <- object$interval
    stopifnot(inherits(object, "bsplines"))

    if (isTRUE(any(object$bsplines)))
      stop("predict.bsplines is not yet implemented for non-proportional hazards setting.")

    coeffBS   <- object$coefficients[1:5]
    nPH       <- object$nPH
    nTD       <- object$nTD
    coeffPred <- object$coefficients[(5 + 5*nTD + 1):(5 + 5*nTD + nPH)]

    m <- eval(object$terms, sys.parent())
    time_name  <- toString(as.name(m[[2]][[2]]))
    event_name <- toString(as.name(m[[2]][[3]]))

    if (is.null(new.data)) new.data <- object$data
    if (is.null(times.pts)) {
      times.pts <- sort(unique(new.data[[time_name]]))
    } else {
      times.pts <- sort(unique(as.numeric(times.pts)))
    }

    if (!(event_name %in% names(new.data))) new.data[[event_name]] <- 0
    if (!(time_name  %in% names(new.data))) new.data[[time_name]]  <- times.pts[1]

    xx <- as.data.frame(model.matrix(object$terms, new.data))[, -1, drop = FALSE]

    if (max(times.pts) > max(int))
      stop("times.pts must be \u2264 max(object$interval).")

    k     <- 3L
    knot  <- c(int[2], int[3])
    delta <- sort(c(rep(c(int[1], int[4]), k), knot))

    Bgrid <- splines::splineDesign(delta, times.pts, ord = k, outer.ok = FALSE)
    haz0  <- as.numeric(exp(Bgrid %*% coeffBS))

    haz_fun_vec <- function(u) {
      B <- splines::splineDesign(delta, u, ord = k, outer.ok = TRUE)
      as.numeric(exp(B %*% coeffBS))
    }

    Cum0 <- vapply(times.pts, function(ti) {
      integrate(haz_fun_vec, lower = 0, upper = ti, rel.tol = 1e-7, abs.tol = 0)$value
    }, numeric(1))
    S0 <- exp(-Cum0)

    if (isTRUE(baseline)) {
      haz <- haz0
      S   <- S0
    } else {
      rrBetaZ <- as.numeric(exp(as.matrix(xx) %*% coeffPred))
      haz <- haz0 * mean(rrBetaZ)
      S   <- vapply(S0, function(s0) mean(s0 ^ rrBetaZ), numeric(1))
    }

    new_predxhaz(
      time     = times.pts,
      hazard   = haz,
      survival = S,
      baseline = "bsplines",
      attrs = list(
        call         = Call,
        pophaz       = object$pophaz,
        interval     = object$interval,
        coefficients = object$coefficients,
        source_class = "xhaz.bsplines"
      )
    )
  }



