% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ui.causal.R
\name{ui.causal}
\alias{ui.causal}
\title{Uncertainty intervals for Average Causal Effects}
\usage{
ui.causal(out.formula, treat.formula, data, rho = c(-0.3, 0.3),
  rho0 = NULL, rho1 = NULL, ACT = FALSE, sand = TRUE, gridn = 21,
  plot = TRUE, rho.plotrange = c(-0.5, 0.5), alpha = 0.05)
}
\arguments{
\item{out.formula}{Formula for the outcome regression models}

\item{treat.formula}{Formula for the propensity score model (regression model for treatment assignment).}

\item{data}{data.frame containing the variables in the formula.}

\item{rho}{Pre-specified interval for \code{rho0} and \code{rho1}.}

\item{rho0}{Pre-specified value of \code{rho0}, if an interval it has to be the same as \code{rho1}.}

\item{rho1}{Pre-specified value of \code{rho1}, if an interval it has to be the same as \code{rho0}.}

\item{ACT}{If TRUE Average Causal effect of the Treated is calculated, if FALSE Average Causal effect is calculated. Default is FALSE.}

\item{sand}{Specifies which estimator of the standard errors should be used for OR, see details.}

\item{gridn}{Number of fixed points within the \code{rho} interval for which sigma0 and sigma1 should be estimated.}

\item{plot}{If TRUE the function runs slightly slower but you will be able to plot your results using \code{\link{plot.uicausal}}.}

\item{rho.plotrange}{an interval larger than \code{rho} for the plot using \code{\link{plot.uicausal}}.}

\item{alpha}{Default 0.05 corresponding to a confidence level of 95 for CI and UI.}
}
\value{
A list containing:
\item{call}{The matched call}
\item{rho0}{The rage of \code{rho0} from which the ui is calculated}
\item{rho1}{If ACT==FALSE,range of \code{rho1} from which the ui is calculated}
\item{out.model0}{Outcome regression model for non-treated.}
\item{out.model1}{Outcome regression model for treated.}
\item{treat.model}{Regression model for treatment mechanism (propensity score).}
\item{sigma0}{Consistent estimate of sigma0 for different values of rho0}
\item{sigma1}{Consistent estimate of sigma1 for different values of rho1}
\item{DR}{DR inference, confidence intervals for different pre-specified values of \code{rho} for the OR estimator, uncertainty interval, coefficient estimates, confounding bias, indentification interval, standard error etc.}
\item{OR}{OR inference, confidence intervals for different pre-specified values of \code{rho} for the OR estimator, uncertainty interval, coefficient estimates, confounding bias, indentification interval, standard error etc.}
}
\description{
This function allows you to derive uncertainty intervals for the average causal effect (ACE) or the average causal effect on the treated (ACT). The function uses a regression imputation estimator and a doubly robust estimator. The uncertainty intervals can be used as a sensitivity analysis to unconfoundedness. Note that \code{rho}=0 render the same results as assuming no unobserved confounding.
}
\details{
In order to visualize the results, you can use \code{\link{plot.uicausal}}. Details about estimators can be found in Genbäck and de Luna (2018)

The standard errors are calculated with the following estimators:

DR ACE - simplified sandwich estimator

DR ACT - sandwich estimator

OR ACE - if sand=TRUE sandwich estimator (default and recommended), if sand=FALSE large sample variance

OR ACT - if sand=TRUE sandwich estimator (default and recommended), if sand=FALSE large sample variance
}
\examples{
library(MASS)
n<-500
delta<-c(-0.3,0.65)
rho<-0.3
X<-cbind(rep(1,n),rnorm(n))
x<-X[,-1]
s0<-2
s1<-3
error<-mvrnorm(n, c(0,0,0), matrix(c(1,0.6,0.9,0.6,4,0.54,0.9,0.54,9), ncol=3))
zstar<-X\%*\%delta+error[,1]
z<- zstar>0
y1<-ifelse(x< (-1),0.2*x-0.1*x^2, ifelse(x< 1,0.3*x, ifelse(x<3,0.4-0.1*x^2,-0.2-0.1*x)))+error[,3]
y0<-ifelse(x<1.5, x-0.4*x^2, ifelse(x<2, -0.15-0.25*x+0.5*x^2, 1.85-0.25*x))+error[,2]
y<-y0
y[z==1]<-y1[z==1]
data<-data.frame(y,z,x)


ui<-ui.causal(y~x, z~x, data=data, rho=c(0,0.3), ACT=FALSE)
ui
plot(ui)
profile(ui)
mean(y1-y0)

ui<-ui.causal(y~x, z~x, data=data, rho=c(0,0.3), ACT=TRUE)
ui
plot(ui)
mean(y1[z==1]-y0[z==1])

}
\references{
Genbäck, M., de Luna, X. (2018). Causal Inference Accounting for Unobserved Confounding after Outcome Regression and Doubly Robust Estimation. \emph{Biometrics}. DOI: 10.1111/biom.13001
}
\author{
Minna Genbäck
}
