% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crosswalk.R
\name{icd_map}
\alias{icd_map}
\title{Translatation of ICD Codes by General Equivalence Mappings (GEMs)}
\usage{
icd_map(dx, from = 9, to = 10, year = 2018,
        method = c("gem", "reverse-gem", "both", "multi-stage"),
        decimal = FALSE, nomatch = c('""', NA),
        output = c("character", "list", "tidy-data"), cache = TRUE, ...)
}
\arguments{
\item{dx}{A character vector representing diagnosis codes.  Each element
of the vector can either represent individual diagnosis code or a set
of diagnosis codes that are concartenated by commas in between.}

\item{from}{A integer value specifying the original code version.
Currently, the available options are \code{9} or \code{10}.}

\item{to}{A integer value specifying the original code version.
Currently, the available options are \code{9} or \code{10}.  If the
input \code{from} and \code{to} are the same, the function will skip
all the translation and return the input \code{dx} with a warning.}

\item{year}{A numeric value specifying the year of the CMS GEMs.  The
currently available options are \code{2017} and \code{2018}.  By
default, 2018 CMS GEMs is used.}

\item{method}{A character string specifying the translateion method.  The
available options are \code{"gem"} for CMS GEM, \code{"reverse-gem"}
for the reverse of CMS GEM, \code{"both"} for both GEM and reverse
GEM, \code{"multi-stage"} for multiple stage procedure.  See Section
Details for more detailed description of the procedure.}

\item{decimal}{A logical value.  If \code{TRUE}, the diagnosis codes would
be returned with decimal points.  The default is \code{FALSE}.}

\item{nomatch}{A character string indicating no translation result can be
found through the specified mapping.  By default, empty strings,
\code{""}, will be used.  Another available option is \code{NA} (or
more specific \code{NA_character_}).  In that case, the code will be
translated to \code{NA_character_} if no translataion result can be
found.}

\item{output}{A character value specifying the format of the output.  The
avaiable options are \code{"character"}, \code{"list"}, and
\code{"tidy-data"}.  By default, option \code{"character"} is used and
results in a character vector that consists of element-wise
concatenatation by commas of all the translated diagnosis codes from
the original codes.  If \code{"list"} is specified, all the translated
codes will not be concartenated and a list of character vectors will
be returned by the function.  Similarly, if \code{"tidy-data"} is
specified, a data frame in a tidy format will be returned.  The first
column of the data frame consists of the original diagnosis codes; the
second column consists of the translated diagnosis codes.}

\item{cache}{A logical value specifying whether to cache all the mappings
for \code{method = "both"} (both CMS GEM and its reverse mapping), and
\code{method = "multi-stage"} (the multiple stage procedure).  If
\code{TRUE} by default, the specified mapping will be generated,
cached and, applied to the translation.  If \code{FALSE}, the CMS GEM
and its reverse mapping will be used for translatation every time
without cache.  It is recommended to set \code{cache = TRUE} for
translation from ICD-9 to ICD-10.  For translation from ICD-10 to
ICD-9, the caching process only takes noticeable time (usually several
minutes at most) for the multi-stage procedure.}

\item{...}{Other arguments for future usage.  A warning will be thrown out
if any argument goes into \code{...} accidentally.}
}
\value{
A character vector of the same length with the input vector will
    be returned by default or if \code{output = "charactor"}.  A list of
    character vectors will be returned if \code{output = "list"}; A data
    frame in tidy-format will be returned if \code{output = "tidy-data"}.
    See argument \code{output} for details.
}
\description{
An open-source implementation in R similar to the Mapping tool developed by
the Agency for Healthcare Research and Quality (AHRQ).
}
\details{
This function aims to efficiently translates the ICD diagnosis codes to the
a different version by the General Equivalence Mappings (GEMs) developed by
the National Center for Health Statistics, Centers for Medicare and Medicaid
Services (CMS), AHIMA, the American Hospital Association, and 3M Health
Information Systems.  The CMS GEMs currently consist of the forward mapping
from ICD-9 codes to ICD-10 codes and the backward mapping from ICD-10 codes
to ICD-9 codes.  In addition to these two mappings, the Agency for
Healthcare Research and Quality (AHRQ) also proposed translation by using
the reverse mappings and multi-stage procedure.

Taking the translation from ICD-9 codes to ICD-10 codes as an example, the
procedure is elaborated as follows: In stage one, the input ICD-9 codes
are mapped to ICD-10 codes using the ICD-9 to ICD-10 forward map as well
as the reverse of the ICD-10 to ICD-9 backward map.  If \code{multiStage =
FALSE}, the procedure will return the translation results from stage one
(and skip the following stages).  Otherwise, the procedure will continue
and become a multiple stage process.  In stage two, the ICD-10 codes
output from the stage one are mapped back to ICD-9 codes using the
backward map as well as the reverse of the forward map; In stage three, it
applies the forward map and reverse-backward map used in stage one again
to the ICD-9 codes from the stage two and return the resulting ICD-10
codes.

The flags of the GEMs are not exported from this function.  For codes with
positive combination flags, the combination of the converted ICD-10 codes is
indicated by the plus sign \code{"+"}.  For example, the ICD-9 code
\code{"24951"} can be translated by 2018 GEMs to ICD-10 code,
\code{"E0839"}, \code{"E0939"}, or one of the codes from \code{("E08311",
"E08319", "E0836", "E09311" "E09319", "E0936")} with \code{"E0865"}.  The
plus sign in the output, such as \code{"E08311+E0865"}, is used to indicate
the combination of \code{"E08311"} and \code{"E0865"}.
}
\examples{
library(touch)

### some random ICD-9 and ICD-10 codes
icd9codes <- c("0011", "001.1", "316", "29383", "E9808", "V90")
icd10codes <- c("F0390", "F0630", "F54", "F30.13", "A010", "M61019")

### forward mapping from ICD-9 to ICD-10
icd_map(icd9codes)
icd_map(icd9codes, decimal = TRUE, nomatch = NA)

### backward mapping from ICD-10 to ICD-9
icd_map(icd10codes, from = 10, to = 9)
icd_map(icd10codes, from = 10, to = 9, nomatch = NA, output = "list")
icd_map(icd10codes, from = 10, to = 9,
        decimal = TRUE, nomatch = NA, output = "tidy")

### reverse-backward mapping from ICD-9 to ICD-10
icd_map(icd9codes, method = "reverse-gem")
icd_map(icd9codes, method = "reverse", decimal = TRUE, nomatch = NA)

### reverse-forward mapping from ICD-10 to ICD-9
icd_map(icd10codes, from = 10, to = 9, method = "reverse-gem")
icd_map(icd10codes, from = 10, to = 9, method = "reverse",
        decimal = TRUE, nomatch = NA)

### forward and reverse-backward mapping from ICD-9 to ICD-10
icd_map(icd9codes, method = "both")
icd_map(icd9codes, method = "both", decimal = TRUE, nomatch = NA)

### backward and reverse-forward mapping from ICD-10 to ICD-9
icd_map(icd10codes, from = 10, to = 9, method = "both")
icd_map(icd10codes, from = 10, to = 9, method = "both",
        decimal = TRUE, nomatch = NA)

### multi-stage process mapping ICD-9 to ICD-10
icd_map(icd9codes, method = "multi-stage")
icd_map(icd9codes, method = "multi-stage", decimal = TRUE, nomatch = NA)

### multi-stage process mapping ICD-10 to ICD-9
icd_map(icd10codes, from = 10, to = 9,
        method = "multi-stage", cache = FALSE)
icd_map(icd10codes, from = 10, to = 9, method = "multi-stage",
        decimal = TRUE, nomatch = NA, cache = FALSE)

### For codes with positive combination flags
icd_map("24951", output = "list")
## where the "+" signs indicate the code combinations
}
\references{
The AHRQ MapIT Automated In-house Stand-alone Mapping Tool. Agency for
Healthcare Research and Quality. 26 March, 2018.
\url{https://qualityindicators.ahrq.gov/resources/toolkits}.
Accessed 5 July, 2022.
}
\seealso{
\code{\link{find_billable}}
}
\author{
Wenjie Wang <wenjie.2.wang@uconn.edu>
}
