% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/travel_times.R
\name{travel_times}
\alias{travel_times}
\title{Calculate shortest travel times from a stop to all reachable stops}
\usage{
travel_times(
  filtered_stop_times,
  stop_name,
  time_range = 3600,
  arrival = FALSE,
  max_transfers = NULL,
  return_coords = FALSE,
  return_DT = FALSE,
  stop_dist_check = 300,
  ...
)
}
\arguments{
\item{filtered_stop_times}{stop_times data.table (with transfers and stops tables as
attributes) created with \code{\link[=filter_stop_times]{filter_stop_times()}} where the
departure or arrival time has been set.}

\item{stop_name}{Stop name for which travel times should be calculated. A vector with
multiple names can be used.}

\item{time_range}{Either a range in seconds or a vector containing the minimal and maximal
departure time (i.e. earliest and latest possible journey departure time)
as seconds or "HH:MM:SS" character. If \code{arrival} is TRUE, \code{time_range}
describes the time window when journeys should end at \code{stop_name}.}

\item{arrival}{If \code{FALSE} (default), all journeys \emph{start} from \code{stop_name}. If
\code{TRUE}, all journeys \emph{end} at \code{stop_name}.}

\item{max_transfers}{The maximum number of transfers. No limit if \code{NULL}}

\item{return_coords}{Returns stop coordinates (lon/lat) as columns if \code{TRUE}. Default is \code{FALSE}.}

\item{return_DT}{travel_times() returns a data.table if \code{TRUE}. Default is \code{FALSE} which
returns a \code{tibble/tbl_df}.}

\item{stop_dist_check}{stop_names are not structured identifiers like stop_ids or
parent_stations, so it is possible that stops with the same name are
far apart from each other.
travel_times() errors if the distance among stop_ids with the same
name is above this threshold (in meters).
Use \code{FALSE} to turn check off. However, it is recommended to
either use \code{\link[=raptor]{raptor()}} or fix the feed (see \code{\link[=cluster_stops]{cluster_stops()}})
in case of warnings.}

\item{...}{ignored}
}
\value{
A table with travel times to/from all stops reachable by \code{stop_name} and their
corresponding journey departure and arrival times.
}
\description{
Function to calculate the shortest travel times from a stop (given by \code{stop_name})
to all other stop_names of a feed. \code{filtered_stop_times} needs to be created before with
\code{\link[=filter_stop_times]{filter_stop_times()}} or \code{\link[=filter_feed_by_date]{filter_feed_by_date()}}.
}
\details{
This function allows easier access to \code{\link[=raptor]{raptor()}} by using stop names instead of ids and
returning shortest travel times by default.

Note however that stop_name might not be a suitable identifier for a feed. It is possible
that multiple stops have the same name while not being related or geographically close to
each other. \code{\link[=stop_group_distances]{stop_group_distances()}} and \code{\link[=cluster_stops]{cluster_stops()}} can help identify and fix
issues with stop_names.
}
\examples{
\donttest{
library(dplyr)

# 1) Calculate travel times from two closely related stops
# The example dataset gtfs_duke has missing times (allowed in gtfs) which is
# why we run interpolate_stop_times beforehand
gtfs = interpolate_stop_times(gtfs_duke)

tts1 = gtfs \%>\%
  filter_feed_by_date("2019-08-26") \%>\%
  travel_times(c("Campus Dr at Arts Annex (WB)", "Campus Dr at Arts Annex (EB)"),
               time_range = c("14:00:00", "15:30:00"))

# you can use either filter_feed_by_date or filter_stop_times to prepare the feed
# the result is the same
tts2 = gtfs \%>\%
 filter_stop_times("2019-08-26", "14:00:00") \%>\%
 travel_times(c("Campus Dr at Arts Annex (WB)", "Campus Dr at Arts Annex (EB)"),
              time_range = 1.5*3600) # 1.5h after 14:00

all(tts1 == tts2)
# It's recommended to store the filtered feed, since it can be time consuming to
# run it for every travel time calculation, see the next example steps

# 2) separate filtering and travel time calculation for a more granular analysis
# stop_names in this feed are not restricted to an area, create clusters of stops to fix
nyc_path <- system.file("extdata", "nyc_subway.zip", package = "tidytransit")
nyc <- read_gtfs(nyc_path)
nyc <- cluster_stops(nyc, group_col = "stop_name", cluster_colname = "stop_name")

# Use journeys departing after 7 AM with arrival time before 9 AM on 26th June
stop_times <- filter_stop_times(nyc, "2018-06-26", 7*3600, 9*3600)

# Calculate travel times from "34 St - Herald Sq"
tts <- travel_times(stop_times, "34 St - Herald Sq", return_coords = TRUE)

# only keep journeys under one hour for plotting
tts <- tts \%>\% filter(travel_time <= 3600)

# travel time to Queensboro Plaza is 810 seconds, 13:30 minutes
tts \%>\%
  filter(to_stop_name == "Queensboro Plaza") \%>\%
  mutate(travel_time = hms::hms(travel_time))

# plot a simple map showing travel times to all reachable stops
# this can be expanded to isochron maps
library(ggplot2)
ggplot(tts) + geom_point(aes(x=to_stop_lon, y=to_stop_lat, color = travel_time))
}
}
