#' @export

#' @title Simulate von Mises process
#' @name rtraj.vmp
#' @description `rtraj.vmp` returns a simulated path of a von Mises process for given parameters
#' @usage rtraj.vmp(n, theta_0, dt, mu, lambda, sigma)
#' @details Let \eqn{\theta_t} evolve according to a von Mises process given by,
#' \deqn{d\theta_t=-\lambda\sin(\theta_t-\mu)dt+\sigma dW_t}
#'
#' We simulate \eqn{\theta_t} by the Euler-Maruyama discretization of the above SDE.
#'
#' @param n number of steps in the simulated path
#' @param dt Time step
#' @param theta_0 initial point
#' @param mu mean parameter
#' @param lambda drift parameter
#' @param sigma volatility parameter
#' @return A vector of length `n` of the simulated path from von Mises process

rtraj.vmp <- function(n, theta_0, dt, mu, lambda, sigma) {
  a <- rep(theta_0, n)
  for (i in 2:n) {
    a[i] <- a[i-1]-lambda*sin(a[i-1]-mu)*dt+sigma*rnorm(1,0,sqrt(dt))
    a[i] <- a[i]%%(2*pi)
    if(a[i]>pi){
      a[i] <- a[i]-2*pi
    }
  }
  return(a)
}
