% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sk_export.R
\name{sk_snap}
\alias{sk_snap}
\title{Snap a set of points to a "sk" grid}
\usage{
sk_snap(from, g = nrow(from), crop_from = FALSE, crop_g = FALSE, quiet = FALSE)
}
\arguments{
\item{from}{matrix, data frame, or points object from \code{sp} or \code{sf}, the source points}

\item{g}{any object accepted or returned by \code{sk}, the destination grid}

\item{crop_from}{logical, indicating to omit points not overlying \code{g}.}

\item{crop_g}{logical, indicating to trim \code{g} to the extent of \code{from}.}

\item{quiet}{logical, suppresses console output}
}
\value{
sk object, a grid containing the snapped points. These are assigned
the corresponding data value in \code{from}, or if  \code{from} has no data, an integer mapping
to the points in \code{from}. Un-mapped grid points are set to NA.
}
\description{
Maps the input points in \code{from} to the closest grid points in the lattice of which
\code{g} is a sub-grid. In cases of duplicate mappings, the function returns the first
matches only.
}
\details{
\code{from} can be a geometry collection from packages \code{sf} or \code{sp}, or a matrix or list
of y and x coordinates. When \code{from} is a matrix, its first two columns should be the
y and x coordinates (in that order), and the (optional) third column should be the
data. When \code{from} is a list, the function expects (two or three) vectors of equal
length, ordered as above.

When \code{from} is a geometry collection with a coordinates reference system (CRS) string,
points are first transformed to the CRS of \code{g}. If one or both of \code{from} and \code{g} are
missing a CRS definition, the function assumes the same one is shared in both.

\code{g} can be a raster geometry object (such as SpatRaster), in which case the function
behaves like \code{terra::rasterize}, or an sk grid object. It can also be a matrix (supplying
dimensions) or a list containing either \code{gdim} or\code{gres}, from which an appropriately
spaced set of grid lines is derived, centered under the bounding box of the points.
If \code{g} is not supplied, it is automatically set to equal \code{nrow(from)}, so that there
there is one grid line along each dimension for each input point.

\code{crop_from} and \code{crop_g} control the extent of the output grid. If both are \code{FALSE}
(the default) the function returns the smallest regular grid containing both \code{g}
and the snapped \code{from} points. If \code{crop_from=TRUE} and \code{crop_g=FALSE} the output
grid will match \code{g} exactly. If \code{crop_from=FALSE} and \code{crop_g=TRUE} the output
grid will include all snapped points, and possibly omit some or all of \code{g}. And if
both are \code{TRUE}, the output grid encloses the intersection of the points with the
bounding box of \code{g}.
}
\examples{

# functions to scale arbitrary inverval to (1, 2,... 100) and make color palettes
num_to_cent = function(x) 1L + floor(99*( x-min(x) ) / diff(range(x)))
my_pal = function(x) grDevices::hcl.colors(x, 'Spectral', rev=TRUE)
my_col = function(x) my_pal(1e2)[ num_to_cent(x) ]

# create a grid object
gdim = c(40, 30)
g = sk(gdim=gdim, gres=1.1)

# randomly position points within bounding box of g
n_pts = 10
from = lapply(g$gyx, function(yx) runif(n_pts, min(yx), max(yx)) )

# translate away from g (no overlap is required)
from[['y']] = from[['y']] + 5
from[['x']] = from[['x']] + 15

# add example data values and plot
from[['z']] = stats::rnorm(length(from[['y']]))
plot(g, reset=FALSE)
graphics::points(from[c('x', 'y')], pch=16, col=my_col(from[['z']]))
graphics::points(from[c('x', 'y')])

# snap only the points overlying the input grid
g_snap = sk_snap(from, g, crop_from=TRUE)
plot(g_snap, col_grid='black', reset=FALSE, leg=FALSE)
graphics::points(from[c('x', 'y')], pch=16, col=my_col(from[['z']]))
graphics::points(from[c('x', 'y')])

# snap all points to grid extension (default settings)
g_snap = sk_snap(from, g, crop_from=FALSE, crop_g=FALSE)
plot(g_snap, col_grid='black', reset=FALSE)
graphics::points(from[c('x', 'y')], pch=16, col=my_col(from[['z']]))
graphics::points(from[c('x', 'y')])

# find smallest subgrid enclosing all snapped grid points
g_snap = sk_snap(from, g, crop_g=TRUE)
plot(g_snap, col_grid='black', reset=FALSE)
graphics::points(from[c('x', 'y')], pch=16, col=my_col(from[['z']]))
graphics::points(from[c('x', 'y')])

# create a new grid of different resolution enclosing all input points
g_snap = sk_snap(from, g=list(gres=c(0.5, 0.5)))
plot(g_snap, reset=FALSE, col_grid='black')
graphics::points(from[c('x', 'y')], pch=16, col=my_col(from[['z']]))
graphics::points(from[c('x', 'y')])

if( requireNamespace('sf') ) {

# a different example, snapping mis-aligned subgrid
g_pts = sk(list(gdim=c(15, 8), gres=1.7), vals=FALSE)
g_pts[['gyx']][['y']] = g_pts[['gyx']][['y']] + 5
g_pts[['gyx']][['x']] = g_pts[['gyx']][['x']] + 5
from = sk_coords(g_pts, out='list')

# convert to sf
eg_sfc = sf::st_geometry(sk_coords(g_pts, out='sf'))
plot(g, reset=FALSE)
plot(eg_sfc, add=TRUE)

# generate example data and plot
eg_sf = sf::st_sf(data.frame(z=stats::rnorm(length(g_pts))), geometry=eg_sfc)
plot(g, reset=FALSE)
plot(eg_sf, pch=16, add=TRUE, pal=my_pal)
plot(eg_sfc, add=TRUE)

# snap points
g_snap = sk_snap(from=eg_sf, g)
plot(g_snap, reset=FALSE, col_grid='black')
plot(eg_sf, pch=16, add=TRUE, pal=my_pal)
plot(eg_sfc, add=TRUE)

# snapping points without data produces the mapping (non-NA values index "from")
g_snap = sk_snap(from=eg_sfc, g)
plot(g_snap, ij=TRUE, reset=FALSE, col_grid='black')
plot(eg_sfc, add=TRUE)

# with crop_g=TRUE)
g_snap = sk_snap(from=eg_sfc, g, crop_g=TRUE)
plot(g_snap, reset=FALSE, col_grid='black')
plot(eg_sfc, add=TRUE)

# test with sp class
eg_sp = as(eg_sf,'Spatial')
g_snap = sk_snap(from=eg_sp, g)
plot(g_snap, reset=FALSE, col_grid='black')
plot(eg_sf, pch=16, add=TRUE, pal=my_pal)
plot(eg_sfc, add=TRUE)

}

}
\seealso{
sk sk_coords

Other sk constructors: 
\code{\link{sk_rescale}()},
\code{\link{sk_sub}()},
\code{\link{sk}()}
}
\concept{sk constructors}
