% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brent.R
\name{brentZero}
\alias{brentZero}
\title{Brent's local root search with extended capabilities}
\usage{
brentZero(
  f,
  interval,
  lower = NA_real_,
  upper = NA_real_,
  f_lower = NULL,
  f_upper = NULL,
  extendInt = "no",
  tol = 1e-08,
  maxiter = 500L,
  trace = 0L
)
}
\arguments{
\item{f}{The function for which the root is sought.}

\item{interval}{A length-2 vector containing the end-points of the search interval}

\item{lower}{Scalar: the lower end point of the search interval. Not necessary if \code{interval} is provided.}

\item{upper}{Scalar: the upper end point of the search interval. Not necessary if \code{interval} is provided.}

\item{f_lower}{Scalar: same as f(upper). Passing this value saves time if f(lower) is slow to compute and is known.}

\item{f_upper}{Scalar: same as f(lower).}

\item{extendInt}{Character:
\describe{
  \item{\code{"no"}}{Do not extend the interval (default).}
  \item{\code{"yes"}}{Attempt to extend both ends until a sign change is found.}
  \item{\code{"upX"}}{Assumes the function is increasing around the root and extends upward if needed.}
  \item{\code{"downX"}}{Assumes the function is decreasing around the root and extends downward if needed.}
  \item{\code{"right"}}{Attempt to extend the upper (right) end until a sign change is found.}
  \item{\code{"left"}}{Attempt to extend the lower (left) end until a sign change is found.}
}
This behavior mirrors that of [uniroot()].}

\item{tol}{Small positive scalar: convergence tolerance. The search stops when the bracket size is smaller than
\code{2 * .Machine$double.eps * abs(x) + tol}, or if the function evaluates to zero at the candidate root.}

\item{maxiter}{Positive integer: the maximum number of iterations before stopping.}

\item{trace}{Integer: 0, 1, or 2. Controls the verbosity of the output.
\code{trace = 0} produces no output, \code{trace = 1} reports the starting and final results,
and \code{trace = 2} provides detailed iteration-level output.}
}
\value{
A list with the following elements:
\describe{
  \item{root}{Location of the root.}
  \item{f.root}{Function value at the root.}
  \item{iter}{Total iteration count used.}
  \item{init.it}{Number of initial \code{extendInt} iterations if there were any; NA otherwise.}
  \item{estim.prec}{Estimate of the final bracket size.}
  \item{exitcode}{0 for success, 1 for maximum initial iteration limit, 2 for maximum main iteration limit.}
}
}
\description{
Brent's local root search with extended capabilities
}
\examples{
f <- function (x, a) x - a
str(uniroot(f, c(0, 1), tol = 0.0001, a = 1/3))
uniroot(function(x) cos(x) - x, lower = -pi, upper = pi, tol = 1e-9)$root

# New capabilities: extending only one end of the interval
f <- function(x) x^2 - 1  # The roots are -1 and 1
brentZero(f, c(2, 3), extendInt = "left")
brentZero(f, c(2, 3), extendInt = "yes")
brentZero(f, c(2, 3), extendInt = "upX")
brentZero(f, c(0, 0.5), extendInt = "downX")  # This one finds the left crossing

# This function is faster than the base R uniroot, and this is the primary
# reason why it was written in C++
system.time(replicate(1000, { shift <- runif(1, 0, 2*pi)
  uniroot(function(x) cos(x+shift) - x, lower = -pi, upper = pi)
}))
system.time(replicate(1000, { shift <- runif(1, 0, 2*pi)
  brentZero(function(x) cos(x+shift) - x, lower = -pi, upper = pi)
}))
# Roughly twice as fast
}
