% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model-cluster.R
\name{sfclust}
\alias{sfclust}
\title{Bayesian spatial functional clustering}
\usage{
sfclust(
  stdata,
  graphdata = NULL,
  stnames = c("geometry", "time"),
  move_prob = c(0.425, 0.425, 0.1, 0.05),
  q = 0.5,
  correction = TRUE,
  niter = 100,
  burnin = 0,
  thin = 1,
  nmessage = 10,
  path_save = NULL,
  nsave = nmessage,
  ...
)
}
\arguments{
\item{stdata}{A stars object containing response variables, covariates, and other necessary data.}

\item{graphdata}{A list containing the initial graph used for the Bayesian model.
It should include components like \code{graph}, \code{mst}, and \code{membership} (default is \code{NULL}).}

\item{stnames}{A character vector specifying the spatio-temporal dimension names of
\code{stdata} that represent spatial geometry and time, respectively (default is
\code{c("geometry", "time")}).}

\item{move_prob}{A numeric vector of probabilities for different types of moves in the MCMC process:
birth, death, change, and hyperparameter moves (default is \code{c(0.425, 0.425, 0.1, 0.05)}).}

\item{q}{A numeric value representing the penalty for the number of clusters (default is \code{0.5}).}

\item{correction}{A logical indicating whether correction to compute the marginal
likelihoods should be applied (default is \code{TRUE}). This depend of the type of
effect inclused in the \code{INLA} model.}

\item{niter}{An integer specifying the number of MCMC iterations to perform (default is \code{100}).}

\item{burnin}{An integer specifying the number of burn-in iterations to discard (default is \code{0}).}

\item{thin}{An integer specifying the thinning interval for recording the results (default is \code{1}).}

\item{nmessage}{An integer specifying how often progress messages should be printed (default is \code{10}).}

\item{path_save}{A character string specifying the file path to save the results (default is \code{NULL}).}

\item{nsave}{An integer specifying the number of iterations between saved results in the chain
(default is \code{nmessage}).}

\item{...}{Additional arguments such as \code{formula}, \code{family}, and others that are passed
to the \code{inla} function.}
}
\value{
An \code{sfclust} object containing two main lists: \code{samples} and \code{clust}.
\itemize{
\item The \code{samples} list includes details from the sampling process, such as:
\itemize{
\item \code{membership}: The cluster membership assignments for each sample.
\item \code{log_marginal_likelihood}: The log marginal likelihood for each sample.
\item \code{move_counts}: The counts of each type of move during the MCMC process.
}
\item The \code{clust} list contains information about the selected clustering, including:
\itemize{
\item \code{id}: The identifier of the selected sample (default is the last sample).
\item \code{membership}: The cluster assignments for the selected sample.
\item \code{models}: The fitted models for each cluster in the selected sample.
}
}
}
\description{
Bayesian detection of neighboring spatial regions with similar functional shapes using
spanning trees and latent Gaussian models. It ensures spatial contiguity in the
clusters, handles a large family of latent Gaussian models supported by \code{inla}, and
allows to work with non-Gaussian likelihoods.
}
\details{
This implementation draws inspiration from the methods described in the paper:
\emph{"Bayesian Clustering of Spatial Functional Data with Application to a Human Mobility
Study During COVID-19"} by Bohai Zhang, Huiyan Sang, Zhao Tang Luo, and Hui Huang,
published in \emph{The Annals of Applied Statistics}, 2023. For further details on the
methodology, please refer to:
\itemize{
\item The paper: \doi{doi:10.1214/22-AOAS1643}
\item Supplementary material: \doi{doi:10.1214/22-AOAS1643SUPPB}
}

The MCMC algorithm in this implementation is largely based on the supplementary
material provided in the paper. However, we have generalized the computation of the
marginal likelihood ratio by leveraging INLA (Integrated Nested Laplace Approximation).
This generalization enables integration over all parameters and hyperparameters,
allowing for inference within a broader family of distribution functions and model
terms, thereby extending the scope and flexibility of the original approach.
Further details of our approach can be found in our paper \emph{"Bayesian spatial functional
data clustering: applications in disease surveillance"} by Ruiman Zhong, Erick A.
Chacón-Montalván, Paula Moraga:
\itemize{
\item The paper: \url{https://arxiv.org/abs/2407.12633}
}
}
\examples{

\donttest{
library(sfclust)

# Clustering with Gaussian data
data(stgaus)
result <- sfclust(stgaus, formula = y ~ f(idt, model = "rw1"),
  niter = 10, nmessage = 1)
print(result)
summary(result)
plot(result)

# Clustering with binomial data
data(stbinom)
result <- sfclust(stbinom, formula = cases ~ poly(time, 2) + f(id),
  family = "binomial", Ntrials = population, niter = 10, nmessage = 1)
print(result)
summary(result)
plot(result)
}

}
\author{
Ruiman Zhong \email{ruiman.zhong@kaust.edu.sa},
Erick A. Chacón-Montalván \email{erick.chaconmontalvan@kaust.edu.sa},
Paula Moraga \email{paula.moraga@kaust.edu.sa}
}
