% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auto_layout.R
\name{auto_layout_mediation}
\alias{auto_layout_mediation}
\title{Set the Layout of a Mediation
Model Automatically}
\usage{
auto_layout_mediation(
  object,
  x = NULL,
  y = NULL,
  exclude = NULL,
  v_pos = c("middle", "lower", "upper"),
  v_preference = c("upper", "lower"),
  output = c("matrix", "xy"),
  update_plot = TRUE
)
}
\arguments{
\item{object}{It can be the output of
\code{\link[lavaan:sem]{lavaan::sem()}} or
\code{\link[lavaan:lavaan]{lavaan::lavaan()}}, or a
\code{lavaan}-class object. The model must
have a \code{beta} matrix of the
structural path. It can also be a
\code{qgraph} object generated by
\code{\link[semPlot:semPaths]{semPlot::semPaths()}}. A `beta``
matrix will be reconstructed from the
graph.}

\item{x}{The variables that will be
treated as (pure) \code{x} variables:
placed on the left of the plot, with
no variables predicting them. If
\code{NULL}, the default, the \code{x}
variable(s) will be identified
automatically.}

\item{y}{The variables that will be
treated as (pure) \code{y} variables:
placed on the right of the plot, with
no variables predicted by them. If
\code{NULL}, the default, the \code{y}
variable(s) will be identified
automatically.}

\item{exclude}{The variables to be
omitted from the plot, typically the
covariates ("control variables") in a
model. If \code{NULL}, the default, all
variables involved in the structural
paths will be used in the plot. It
is possible to exclude \code{y}-variables.
However, excluding mediators is not
allowed.}

\item{v_pos}{How the mediators are to
be positioned vertically in the
first pass. If
\code{"middle"}, the function will try to
position them close to the center of
the plot. If \code{"lower"}, it will try
to position them to the lower part of
the plot. If \code{"upper"}, it will try
to position them to the upper part of
the plot.}

\item{v_preference}{The preference in
shifting the mediators upward
(\code{"upper"}) or downward (\code{"lower"})
in the second pass to avoid blocking
or overlapping with any paths in the
models. It is used only when\code{v_pos}
is \code{"middle"}. If \code{v_pos} is
\code{"lower"}, then \code{v_preference} will
be forced to be \verb{"lower". If }v_pos\code{is}"upper"\verb{, then }v_preference\verb{will be forced to be}"upper".}

\item{output}{The format of the
output, used if \code{update_plot} is
\code{FALSE}. If \code{"matrix"}, the output is
a two-dimension character matrix with
the names of the variables. If
\code{"xy"}, the output is a two-column
matrix of the relatived x- and
y-positions of each variables.}

\item{update_plot}{Logical. Used
if \code{object} is a \code{qgraph} object. If
\code{TRUE}, the function returns a
modified \code{qgraph} object with the
new layout. If \code{FALSE}}
}
\value{
If \code{object} is a \code{lavaan}-class
object, or if \code{update_plot} is \code{FALSE},
it returns
a two-dimension layout matrix of the
position of the nodes, or a
two-column matrix of the x-y positions
of the nodes, depending on the
argument \code{output}.

If \code{object} is a \code{qgraph} object
and \code{update_plot} is \code{TRUE}, it
returns a \code{qgraph} object with the
the modified layout.
}
\description{
Set the layout of
variables in a mediation model in the
typical left-to-right style
automatically.
}
\details{
Typically, a path model with some
\code{x} variables, some \code{y} variables,
and some mediators are drawn from
left to right. This function tries
to generate the layout matrix
automatically, meeting the following
requirements:
\itemize{
\item The predictor(s), \code{x} variables(x),
is/are placed to the left.
\item The outcome variable(s), \code{y}
variable(s), is/are placed to the
right.
\item The mediator(s) are positioned
between \code{x} variable(s) and \code{y}
variable(s) such that all paths
point to the right. That is,
no vertical path.
\item The vertical position(s) of the
mediator(s) will be adjusted such
that no path passes through a
mediator. That is, all paths are
visible and not blocked by any
mediator.
}
}
\examples{

library(lavaan)
library(semPlot)

# Create a dummy dataset
mod_pa <-
"
m11 ~ c1 + x1
m21 ~ c2 + m11
m2 ~ m11 + c3
m22 ~ m11 + c3
y ~ m2 + m21 + m22 + x1
"
fit <- lavaan::sem(
          mod_pa,
          do.fit = FALSE
        )
dat <- simulateData(
          parameterTable(fit),
          sample.nobs = 500,
          seed = 1234
        )
fit <- lavaan::sem(
          mod_pa,
          dat
        )

# Set the layout
m <- auto_layout_mediation(
        fit,
        exclude = c("c1", "c2", "c3")
      )
pm <- semPlotModel(fit) |> drop_nodes(c("c1", "c2", "c3"))
semPaths(
          pm,
          whatLabels = "est",
          layout = m
        )

# v_pos = "lower"
m <- auto_layout_mediation(
        fit,
        exclude = c("c1", "c2", "c3"),
        v_pos = "lower"
      )
pm <- semPlotModel(fit) |> drop_nodes(c("c1", "c2", "c3"))
p0 <- semPaths(
          pm,
          whatLabels = "est",
          layout = m
        )

# v_pos = "upper"
m <- auto_layout_mediation(
        fit,
        exclude = c("c1", "c2", "c3"),
        v_pos = "upper"
      )
pm <- semPlotModel(fit) |> drop_nodes(c("c1", "c2", "c3"))
p0 <- semPaths(
          pm,
          whatLabels = "est",
          layout = m
        )

# Can modify a qgraph

pm <- semPlotModel(fit) |> drop_nodes(c("c1", "c2", "c3"))
p <- semPaths(
          pm,
          whatLabels = "est"
        )
p2 <- auto_layout_mediation(p)
plot(p2)



}
\seealso{
\code{\link[=set_sem_layout]{set_sem_layout()}}. The
output of \code{\link[=auto_layout_mediation]{auto_layout_mediation()}}
can be used by \code{\link[=set_sem_layout]{set_sem_layout()}}.
}
