\name{Example1}
\alias{Example1}
\title{
Subgroup search in a clinical trial with a continuous endpoint 
}
\description{
Consider a clinical trial that was conducted to evaluate the efficacy and safety of an experimental treatment compared to placebo. The primary endpoint is a continuous endpoint. The trial data can be found in the \link{continuous} data set. This data set includes five biomarkers (four numeric biomarkers and one nominal biomarker) as well as several covariates that can be included in the primary analysis model. The trial's sponsor is interested in identifying a set of promising subgroups with enhanced treatment effect. Subgroup search will be performed using the basic SIDES procedure and two SIDEScreen procedures (Fixed and Adaptive SIDEScreen procedures).\cr
The endpoint parameters will be defined as follows. First of all, the name of the outcome variable needs to be specified, i.e., \code{outcome_variable = "outcome"}. Since the endpoint is continuous, the \code{type} parameter is \code{"continuous"}, and further the \code{direction} parameter is set to 1 since a higher value of the primary endpoint indicates beneficial effect. \cr
Two sets of endpoint parameters will be considered to implement a simple evaluation of the treatment effect based on the two-sample t-test and a more advanced analysis based on an ANCOVA model with an adjustment for important prognostic covariates:
\itemize{
\item Analysis strategy 1: The \code{analysis_method} parameter is set to \code{"T-test"}.
\item Analysis strategy 2: The \code{analysis_method} parameter is set to \code{"ANCOVA"}. The covariates to be included in the model need to be defined using the \code{cont_covariates} and \code{class_covariates} arguments. Suppose that the ANCOVA model will account for two continuous covariates (cont1, cont2) and one class/categorical covariate (class1), then \code{cont_covariates} is set to \code{"cont1, cont2"} and \code{class_covariates} is set to \code{"class1"}.
}
The data set parameters will be specified as follows:
\itemize{
\item The data set's name (\code{data_set}) is \code{continuous}.
\item The treatment variable's name (\code{treatment_variable_name}) is \code{"treatment"}.
\item The value of the treatment variable that defines the control arm \cr (\code{treatment_variable_control_value}) is \code{"0"}.
\item The list of candidate biomarkers to be included in the subgroup search (\code{biomarker_names}) is \code{c("biomarker1", "biomarker2", "biomarker3", "biomarker4", "biomarker5")}.
\item The list of biomarker types (\code{biomarker_types}) is \code{c(rep("numeric", 4), "nominal")}.
}
Finally, the following algorithm parameters will be used in the subgroup search procedures:
\itemize{
\item Search depth (\code{depth}) is 2, which means that patient subgroups will be defined in terms of one or two biomarkers. Note that this is the default value of this parameter and thus it does not need to be explicitly defined.
\item Search width (\code{width}) is 2, i.e., only up to two best child subgroups will be retained for each parent group. This is the default value of this parameter and thus it does not need to be explicitly defined.
\item Child-to-parent ratio (\code{gamma}) equals 1 for each search level. This is the default value of this parameter and thus it does not need to be explicitly defined.
\item Minimum total number of patients in a promising subgroup (\code{min_subgroup_size}) is 60.
\item Minimum number of unique values for continuous biomarkers for applying a percentile transformation (\code{nperc}) is 20, i.e., a percentile transformation will not be applied to a continuous biomarker if there are few than 20 unique values. This is the default value of this parameter and thus it does not need to be explicitly defined.
\item Number of permutations to compute multiplicity-adjusted treatment effect p-values within promising subgroups (\code{n_perms_mult_adjust}) is 10. 
\item Number of processor cores to be used in parallel calculations (\code{ncores}) is 1. 
}
The subgroup search will be performed using the basic SIDES procedure as well as two SIDEScreen procedures by calling the \link{SubgroupSearch} function and a comprehensive summary of subgroup search results will be generated by calling the \link{GenerateReport} function.
}

\seealso{
  \code{\link{Example2}, \link{Example3}}
}
\examples{
  \donttest{
    ##############################################################################

    # Primary endpoint parameters

    # Analysis strategy 1: Analysis of the continuous endpoint without 
    # accounting for any covariates
    endpoint_parameters = list(outcome_variable = "outcome", 
      type = "continuous",
      label = "Outcome", 
      analysis_method = "T-test", 
      direction = 1)

    # Analysis strategy 2: Analysis of the continuous endpoint using an ANCOVA 
    # model that accounts for two continuous covariates (cont1, cont2) and 
    # one class/categorical covariate (class1)
    endpoint_parameters = list(outcome_variable = "outcome", 
      type = "continuous",
      label = "Outcome", 
      analysis_method = "ANCOVA", 
      cont_covariates = "cont1, cont2", 
      class_covariates = "class1", 
      direction = 1)

    ##############################################################################

    # Data set parameters

    # Set of candidate biomarkers
    biomarker_names = c("biomarker1", "biomarker2", 
                        "biomarker3", "biomarker4", 
                        "biomarker5")

    # Biomarker type 
    biomarker_types = c(rep("numeric", 4), "nominal")

    # Data set parameters
    data_set_parameters = list(data_set = continuous,
      treatment_variable_name = "treatment",
      treatment_variable_control_value = "0",
      biomarker_names = biomarker_names,
      biomarker_types = biomarker_types)

    ##############################################################################

    # Algorithm parameters for the basic SIDES procedure

    # Algorithm
    subgroup_search_algorithm = "SIDES procedure"

    # Number of permutations to compute multiplicity-adjusted treatment 
    # effect p-values within promising subgroups
    n_perms_mult_adjust = 10

    # Number of processor cores (use less or equal number of CPU cores on the current host)
    ncores = 1

    # Default values for the search depth (2), search width (2), 
    # maximum number of unique values for continuous biomarkers (20)

    # Algorithm parameters
    algorithm_parameters = list(
      n_perms_mult_adjust = n_perms_mult_adjust,
      min_subgroup_size = 60,
      subgroup_search_algorithm = subgroup_search_algorithm,
      ncores = ncores,
      random_seed = 3011)

    # Perform subgroup search

    # List of all parameters
    parameters = list(endpoint_parameters = endpoint_parameters,
      data_set_parameters = data_set_parameters,
      algorithm_parameters = algorithm_parameters)

    results = SubgroupSearch(parameters)

    # Simple summary of subgroup search results
    results

    # Generate a detailed Word-based report with a summary of subgroup search results
    report_information = GenerateReport(results,
      report_title = "Subgroup search report", 
      report_filename = tempfile(
        "Continuous endpoint (SIDES).docx", 
        fileext=".docx"
      )
    )

    ##############################################################################

    # Algorithm parameters for the Fixed SIDEScreen procedure

    # Algorithm
    subgroup_search_algorithm = "Fixed SIDEScreen procedure"

    # Number of permutations to compute multiplicity-adjusted treatment 
    # effect p-values within promising subgroups
    n_perms_mult_adjust = 10

    # Number of processor cores (use less or equal number of CPU cores on the current host)
    ncores = 1

    # Number of biomarkers selected for the second stage in the Fixed SIDEScreen algorithm
    n_top_biomarkers = 3

    # Default values for the search depth (2), search width (2), 
    # maximum number of unique values for continuous biomarkers (20)

    # Algorithm parameters
    algorithm_parameters = list(
      n_perms_mult_adjust = n_perms_mult_adjust,
      min_subgroup_size = 60,
      subgroup_search_algorithm = subgroup_search_algorithm,
      ncores = ncores,
      n_top_biomarkers = n_top_biomarkers,
      random_seed = 3011)

    # Perform subgroup search

    # List of all parameters
    parameters = list(endpoint_parameters = endpoint_parameters,
      data_set_parameters = data_set_parameters,
      algorithm_parameters = algorithm_parameters)

    results = SubgroupSearch(parameters)

    # Simple summary of subgroup search results
    results

    # Generate a detailed Word-based report with a summary of subgroup search results
    report_information = GenerateReport(results,
      report_title = "Subgroup search report", 
      report_filename = tempfile(
        "Continuous endpoint (Fixed SIDEScreen).docx", 
        fileext=".docx"
      )
    )

    ##############################################################################

    # Algorithm parameters for the Adaptive SIDEScreen procedure

    # Algorithm
    subgroup_search_algorithm = "Adaptive SIDEScreen procedure"

    # Number of permutations to compute multiplicity-adjusted treatment 
    # effect p-values within promising subgroups
    n_perms_mult_adjust = 10

    # Number of processor cores (use less or equal number of CPU cores on the current host)
    ncores = 1

    # Multiplier for selecting biomarkers for the second stage 
    # in the Adaptive SIDEScreen algorithm
    multiplier = 1

    # Number of permutations for computing the null distribution
    # of the maximum VI score in the Adaptive SIDEScreen algorithm
    n_perms_vi_score = 100

    # Default values for the search depth (2), search width (2), 
    # maximum number of unique values for continuous biomarkers (20)

    # Algorithm parameters
    algorithm_parameters = list(
      n_perms_mult_adjust = n_perms_mult_adjust,
      min_subgroup_size = 60,
      subgroup_search_algorithm = subgroup_search_algorithm,
      ncores = ncores,
      multiplier = multiplier,
      n_perms_vi_score = n_perms_vi_score,
      random_seed = 3011)

    # Perform subgroup search

    # List of all parameters
    parameters = list(endpoint_parameters = endpoint_parameters,
      data_set_parameters = data_set_parameters,
      algorithm_parameters = algorithm_parameters)

    results = SubgroupSearch(parameters)

    # Simple summary of subgroup search results
    results

    # Generate a detailed Word-based report with a summary of subgroup search results
    GenerateReport(results,
      report_title = "Subgroup search report", 
      report_filename = tempfile(
        "Continuous endpoint (Adaptive SIDEScreen).docx", 
        fileext=".docx"
      )
    )
  }
}
