\name{bootstrap_test}
\alias{bootstrap_test}
\title{Bootstrap Test for Comparing Estimates in Stochastic Frontier Models}
\usage{
bootstrap_test(formula, data = NULL, alpha0, alpha1, B = 99)
}
\arguments{
  \item{formula}{A symbolic description of the model to be estimated,
    specified using the standard R formula syntax (e.g., y ~ x1 + x2).}

  \item{data}{A data frame containing the variables in the model.}

  \item{alpha0}{First value of \eqn{\alpha}. The bootstrap samples are
    generated using the MDPD estimates with \eqn{\alpha=\alpha_0}. }

  \item{alpha1}{Second value of \eqn{\alpha}.}

  \item{B}{A numeric value specifying the number of bootstrap
    replications. The default is 99.}
}
\value{
  A numeric. \emph{p}-value of the bootstrap test.
}
\description{
  This function performs a bootstrap test to assess the closeness of two
  MDPD estimates at different values of \eqn{\alpha}. This test is based
  on the observation that outliers particularly affect the estimation of
  \eqn{\sigma^2} and \eqn{\lambda}. Accordingly, the bootstrap test is
  conducted using the following similarity measure:
  
  \deqn{
  sim(\hat\theta_0,\hat\theta_1)= \frac{|\hat \sigma^2_1-\hat\sigma^2_0|}{\hat\sigma^2_0}
  +\frac{|\hat\lambda_1-\hat\lambda_0|}{\hat\lambda_0},}{See the pdf version of help files.}

  where \eqn{\hat\theta_0} and \eqn{\hat\theta_1} are the MDPD estimates
  corresponding to \eqn{\alpha = \alpha_0} and \eqn{\alpha = \alpha_1},
  respectively. If the two MDPD estimates are close,
  \eqn{sim(\hat\theta_0,\hat\theta_1)} will be close to zero. We note
  that this similarity measure differs from the one used in Song et
  al. (2017). Apart from this, the bootstrap procedure follows the same
  steps described in Song et al. (2017). A low \emph{p}-value indicates
  that the two estimates are significantly different. Note that this
  test may require significant computational time, as it involves
  numerous estimation procedures.
}
\examples{
\donttest{
## Example using the 'riceProdPhil' dataset from the `frontier` package
library(frontier)
data(riceProdPhil)

my.model <- log(PROD) ~ log(AREA) + log(LABOR) + log(NPK) + log(OTHER)


## Evaluate the closeness of ML estimates (alpha = 0) and
## MDPD estimates with alpha = 0.5.
bootstrap_test(my.model, data = riceProdPhil, alpha0=0.5, alpha1=0)


## Data with a single outlying observation
riceProdPhil2 <- riceProdPhil
riceProdPhil3 <- riceProdPhil

idx <- which.max(riceProdPhil$PROD)
riceProdPhil2$PROD[idx] <- riceProdPhil$PROD[idx]*10
riceProdPhil3$PROD[idx] <- riceProdPhil$PROD[idx]/100


## Evaluate the closeness of ML estimates (alpha = 0) and
## MDPD estimates with alpha = 0.5.
bootstrap_test( my.model, data = riceProdPhil2, alpha0=0.5, alpha1=0)
bootstrap_test( my.model, data = riceProdPhil3, alpha0=0.5, alpha1=0)
}
}
