% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/1-0-0rdlearn.R
\name{rdlearn}
\alias{rdlearn}
\title{Safe Policy Learning for Regression Discontinuity Design with Multiple
Cutoffs}
\usage{
rdlearn(
  y,
  x,
  c,
  group_name = NULL,
  data,
  fold = 10,
  M = 1,
  cost = 0,
  trace = TRUE
)
}
\arguments{
\item{y}{A character string specifying the name of column containing the outcome variable.}

\item{x}{A character string specifying the name of column containing the running variable.}

\item{c}{A character string specifying the name of column containing the cutoff variable.}

\item{group_name}{A character string specifying the name of the column
containing group names (e.g., department names) for each cutoff. If not
provided, the groups are assigned names "Group 1", "Group 2", ... in
ascending order of cutoff values.}

\item{data}{A data frame containing all required variables.}

\item{fold}{The number of folds for cross-fitting. Default is 10.}

\item{M}{A numeric value or vector specifying the multiplicative smoothness
factor(s) for sensitivity analysis. Default is 1.}

\item{cost}{A numeric value or vector specifying the cost of treatment for
calculating regret. This cost should be scaled by the range of the outcome
variable Y. Default is 0.}

\item{trace}{A logical value that controls whether to display the progress.
If set to TRUE, the progress will be printed. The default value is TRUE.}
}
\value{
An object of class \code{rdlearn}, which is a list containing the
  following components:
  \describe{
    \item{call}{The original function call.}
    \item{var_names}{A list of variable names for the outcome, running variable, and cutoff.}
    \item{org_cut}{A vector of original cutoff values.}
    \item{safe_cut}{A data frame containing the obtained new treatment assignment cutoffs.}
    \item{sample}{The total sample size.}
    \item{num_group}{The number of groups.}
    \item{group_name}{A vector of group names.}
    \item{cross_fit_output}{The intermediate output of the cross-fitting procedure.}
    \item{dif_lip_output}{The intermediate output of the cross-group differences and the smoothness parameters}
    \item{distance}{A numeric vector containing the measures of difference between safe cutoffs and original cutoffs}
    \item{rdestimates}{A data frame containing the result of \code{rdesimate} such as causal effect estimates.}
    \item{temp_reg_df}{A data frame containing the regrets of every alternative cutoff.}
  }
}
\description{
The \code{rdlearn} function implements safe policy learning under a
regression discontinuity design with multiple cutoffs. The resulting new
treatment assignment rules (cutoffs) are guaranteed to yield no worse overall
outcomes than the existing cutoffs.
}
\details{
Regarding the detail of the algorithm, please refer to Zhang et al. (2022) "4 Empirical policy
learning" and "A.2 A double robust estimator for heterogeneous cross-group
differences".
}
\examples{
# Simulation Data B from Appendix D of Zhang et al. (2022)
set.seed(1)
n <- 300
X <- runif(n, -1000, -1)
G <- 2 * as.numeric(
I(0.01 * X + 5 + rnorm(n, sd = 10) > 0)
) +
as.numeric(
I(0.01 * X + 5 + rnorm(n, sd = 10) <= 0)
)
c1 <- -850
c0 <- -571
C <- ifelse(G == 1, c1, c0)
D <- as.numeric(X >= C)
coef0 <- c(-1.992230e+00, -1.004582e-02, -1.203897e-05, -4.587072e-09)
coef1 <- c(9.584361e-01, 5.308251e-04, 1.103375e-06, 1.146033e-09)
Px <- poly(X, degree = 3, raw = TRUE)
# Px = poly(X-735.4334-c1,degree=3,raw=TRUE) for Simulation A
Px <- cbind(rep(1, nrow(Px)), Px)
EY0 <- Px \%*\% coef0
EY1 <- Px \%*\% coef1
d <- 0.2 + exp(0.01 * X) * (1 - G) + 0.3 * (1 - D)
Y <- EY0 * (1 - D) + EY1 * D - d * as.numeric(I(G == 1)) + rnorm(n, sd = 0.3)

simdata_B_demo <- data.frame(Y,X,C)

# Learn new treatment assignment cutoffs
rdlearn_result <- rdlearn(
  y = "Y", x = "X", c = "C", data = simdata_B_demo,
  fold = 2, M = 0, cost = 0
)

# Summarise the learned policies
summary(rdlearn_result)

# Visualize the learned policies
plot(rdlearn_result, opt = "dif")
# The learned cutoff for Group 1 is the same as the baseline cutoff, because
# the baseline cutoff is set to equal to oracle cutoff in this simulation.

# Implement sensitivity analysis
sens_result <- sens(rdlearn_result, M = 1, cost = 0)
plot(sens_result, opt = "dif")
}
