#include "color_grid.h"
#include "cvd.h"
#include "farthest_points.h"
#include "palettes.h"
#include "palettes_data.h"
#include "validation.h"
#include <cassert>
#include <qualpal/colors.h>
#include <qualpal/qualpal.h>
#include <stdexcept>

namespace qualpal {

Qualpal&
Qualpal::setInputRGB(const std::vector<colors::RGB>& colors)
{
  this->rgb_colors_in = colors;
  this->mode = Mode::RGB;
  return *this;
}

Qualpal&
Qualpal::setInputHex(const std::vector<std::string>& hex_colors)
{
  for (const auto& color : hex_colors) {
    if (!isValidHexColor(color)) {
      throw std::invalid_argument("Invalid hex color: " + color +
                                  ". Expected format: #RRGGBB or #RGB");
    }
  }
  this->hex_colors = hex_colors;
  this->mode = Mode::HEX;
  return *this;
}

Qualpal&
Qualpal::setInputPalette(const std::string& palette)
{
  validatePalette(palette);
  this->palette = palette;
  this->mode = Mode::PALETTE;
  return *this;
}

Qualpal&
Qualpal::setInputColorspace(const std::array<double, 2>& h_lim,
                            const std::array<double, 2>& s_or_c_lim,
                            const std::array<double, 2>& l_lim,
                            ColorspaceType space)
{
  ColorspaceRegion region{ h_lim, s_or_c_lim, l_lim };
  return setInputColorspaceRegions({ region }, space);
}

Qualpal&
Qualpal::setInputColorspaceRegions(const std::vector<ColorspaceRegion>& regions,
                                   ColorspaceType space)
{
  if (regions.empty()) {
    throw std::invalid_argument("At least one colorspace region is required");
  }

  // Validate each region
  for (const auto& region : regions) {
    if (space == ColorspaceType::HSL) {
      if (region.h_lim[0] < -360 || region.h_lim[1] > 360) {
        throw std::invalid_argument("Hue must be between -360 and 360");
      }

      if (region.h_lim[1] - region.h_lim[0] > 360) {
        throw std::invalid_argument("Hue range must be less than 360");
      }

      if (region.s_or_c_lim[0] < 0 || region.s_or_c_lim[1] > 1) {
        throw std::invalid_argument(
          "Saturation/chroma must be between 0 and 1");
      }

      if (region.l_lim[0] < 0 || region.l_lim[1] > 1) {
        throw std::invalid_argument("Lightness must be between 0 and 1");
      }
    } else if (space == ColorspaceType::LCHab) {
      if (region.h_lim[0] < -360 || region.h_lim[1] > 360) {
        throw std::invalid_argument("Hue must be between -360 and 360");
      }

      if (region.s_or_c_lim[0] < 0) {
        throw std::invalid_argument("Chroma must be non-negative");
      }

      if (region.l_lim[0] < 0 || region.l_lim[1] > 100) {
        throw std::invalid_argument("Lightness must be between 0 and 100");
      }
    }
  }

  this->colorspace_regions = regions;
  this->colorspace_input = space;
  this->mode = Mode::COLORSPACE;
  return *this;
}

Qualpal&
Qualpal::setCvd(const std::map<std::string, double>& cvd_params)
{
  for (const auto& [cvd_type, cvd_severity] : cvd_params) {
    if (cvd_severity > 1.0 || cvd_severity < 0.0) {
      throw std::invalid_argument("cvd_severity must be between 0 and 1");
    }
    if (cvd_type != "protan" && cvd_type != "deutan" && cvd_type != "tritan") {
      throw std::invalid_argument(
        "Invalid CVD type: " + cvd_type +
        ". Supported types are: protan, deutan, tritan.");
    }
  }
  this->cvd = cvd_params;
  return *this;
}

Qualpal&
Qualpal::setBackground(const colors::RGB& bg_color)
{
  this->bg = bg_color;
  return *this;
}

Qualpal&
Qualpal::setMetric(metrics::MetricType metric)
{
  this->metric = metric;
  return *this;
}

Qualpal&
Qualpal::setMemoryLimit(double gb)
{
  if (gb <= 0) {
    throw std::invalid_argument("Memory limit must be greater than 0");
  }
  this->max_memory = gb;
  return *this;
}

Qualpal&
Qualpal::setColorspaceSize(std::size_t n_points)
{
  if (n_points <= 0) {
    throw std::invalid_argument("Number of points must be greater than 0");
  }
  this->n_points = n_points;
  return *this;
}

Qualpal&
Qualpal::setWhitePoint(WhitePoint wp)
{
  this->white_point = whitePointToXYZ(wp);
  return *this;
}

Qualpal&
Qualpal::setWhitePoint(const std::array<double, 3>& white_point)
{
  this->white_point = white_point;
  return *this;
}

std::vector<colors::RGB>
Qualpal::selectColors(std::size_t n,
                      const std::vector<colors::RGB>& fixed_palette)
{
  switch (mode) {
    case Mode::RGB:
      break;
    case Mode::HEX:
      rgb_colors_in.clear();
      rgb_colors_in.reserve(n_points);
      for (const auto& hex : hex_colors) {
        rgb_colors_in.emplace_back(hex);
      }
      break;
    case Mode::PALETTE:
      rgb_colors_in.clear();
      rgb_colors_in.reserve(n_points);
      for (const auto& hex : getPalette(palette)) {
        rgb_colors_in.emplace_back(hex);
      }
      break;
    case Mode::COLORSPACE: {
      rgb_colors_in.clear();

      // Calculate points per region (distribute evenly)
      std::size_t points_per_region = n_points / colorspace_regions.size();
      std::size_t remainder = n_points % colorspace_regions.size();

      rgb_colors_in.reserve(n_points);

      for (std::size_t i = 0; i < colorspace_regions.size(); ++i) {
        const auto& region = colorspace_regions[i];
        // Add one extra point to first 'remainder' regions to distribute evenly
        std::size_t region_points = points_per_region + (i < remainder ? 1 : 0);

        if (colorspace_input == ColorspaceType::HSL) {
          for (const auto& hsl : colorGrid<colors::HSL>(region.h_lim,
                                                        region.s_or_c_lim,
                                                        region.l_lim,
                                                        region_points)) {
            rgb_colors_in.emplace_back(hsl);
          }
        } else if (colorspace_input == ColorspaceType::LCHab) {
          for (const auto& lch : colorGrid<colors::LCHab>(region.h_lim,
                                                          region.s_or_c_lim,
                                                          region.l_lim,
                                                          region_points)) {
            rgb_colors_in.emplace_back(lch);
          }
        }
      }
      break;
    }
    case Mode::NONE:
      throw std::runtime_error("No input source configured.");
  }

  if (rgb_colors_in.empty()) {
    throw std::runtime_error("No input colors provided.");
  }

  std::size_t n_fixed = fixed_palette.size();

  if (n < n_fixed) {
    throw std::invalid_argument(
      "Requested palette size is less than the size of the existing palette.");
  }

  if (n < n_fixed) {
    throw std::invalid_argument("Number of new colors to add is negative.");
  }

  std::size_t n_new = n - n_fixed;

  if (rgb_colors_in.size() < n_new) {
    throw std::invalid_argument(
      "Requested number of colors exceeds input size");
  }

  bool has_bg = bg.has_value();

  std::vector<colors::RGB> rgb_colors;
  rgb_colors.reserve(fixed_palette.size() + rgb_colors_in.size() +
                     (has_bg ? 1 : 0));
  rgb_colors.insert(
    rgb_colors.end(), fixed_palette.begin(), fixed_palette.end());
  rgb_colors.insert(
    rgb_colors.end(), rgb_colors_in.begin(), rgb_colors_in.end());
  if (has_bg) {
    rgb_colors.push_back(*bg);
  }

  // Convert colors to XYZ for distance calculations
  std::vector<colors::XYZ> xyz_colors;
  xyz_colors.reserve(rgb_colors.size());
  for (const auto& c : rgb_colors) {
    xyz_colors.emplace_back(c);
  }

  // Select new colors (CVD-aware if CVD parameters are set)
  auto ind = farthestPoints(
    n, xyz_colors, metric, has_bg, n_fixed, max_memory, white_point, cvd);

  // Output: fixed_palette + selected new colors
  std::vector<colors::RGB> result;
  for (const auto& i : ind) {
    result.emplace_back(rgb_colors[i]);
  }

  return result;
}

std::vector<colors::RGB>
Qualpal::generate(std::size_t n)
{
  return selectColors(n);
}

std::vector<colors::RGB>
Qualpal::extend(const std::vector<colors::RGB>& palette, std::size_t n)
{
  return selectColors(n, palette);
}

} // namespace qualpal
