% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solvers.R
\name{solvers}
\alias{solvers}
\title{Add solvers}
\description{
Specify the software and settings used to solve a conservation planning
problem. By default, the best available
software currently installed on the system will be used.
For information on the performance of different solvers,
please see Hanson \emph{et al.} (2025) and Schuster \emph{et al.} (2020) for
benchmarks comparing run time and solution quality of these solvers.
}
\details{
The following functions can be used to add a solver to a
conservation planning \code{\link[=problem]{problem()}}. Note that if multiple
of these functions are added to a \code{\link[=problem]{problem()}}, then only the last
function added will be used.

\describe{

\item{\code{\link[=add_default_solver]{add_default_solver()}}}{
This solver uses the best software currently installed on the system.
}

\item{\code{\link[=add_gurobi_solver]{add_gurobi_solver()}}}{
\href{https://www.gurobi.com/}{\emph{Gurobi}}
is a state-of-the-art commercial optimization software with an R package
interface. We recommend using this solver if at all possible.
It is by far the fastest of the solvers available for
generating prioritizations, however, it is not freely available. That
said, licenses are available to academics at no cost. The
\pkg{gurobi} package is distributed with the \emph{Gurobi} software
suite. This solver uses the \pkg{gurobi} package to solve problems.
}

\item{\code{\link[=add_cplex_solver]{add_cplex_solver()}}}{
\href{https://www.ibm.com/products/ilog-cplex-optimization-studio/cplex-optimizer}{\emph{IBM CPLEX}}
is a commercial optimization software. It is faster than the open
source solvers available for generating prioritizations, however, it
is not freely available.
Similar to the \href{https://www.gurobi.com/}{\emph{Gurobi}}
software, licenses are available to academics at no cost.
This solver uses the \pkg{cplexAPI} package to solve problems using
\emph{IBM CPLEX}.
}

\item{\code{\link[=add_cbc_solver]{add_cbc_solver()}}}{
\href{https://github.com/coin-or/Cbc}{\emph{CBC}} is an
open-source mixed integer programming solver that is part of the
Computational Infrastructure for Operations Research (COIN-OR) project.
Preliminary benchmarks indicate that it is the fastest open source
solver currently supported.
We recommend using this solver if both \emph{Gurobi} and \emph{IBM CPLEX} are
unavailable.
It requires the \pkg{rcbc} package, which is currently only available on
\href{https://github.com/dirkschumacher/rcbc}{GitHub}.
}

\item{\code{\link[=add_highs_solver]{add_highs_solver()}}}{
\href{https://highs.dev/}{\emph{HiGHS}} is an open
source optimization software. Although this solver can have
comparable performance to the \emph{CBC} solver for particular problems and is
generally faster than the \emph{SYMPHONY} based solvers (see below), it
sometimes can take much longer than the \emph{CBC} solver for particular
problems.
}

\item{\code{\link[=add_lpsymphony_solver]{add_lpsymphony_solver()}}}{
\href{https://github.com/coin-or/SYMPHONY}{\emph{SYMPHONY}} is an
open-source mixed integer programming solver that is also part of the
COIN-OR project. Although both \emph{SYMPHONY} and \emph{CBC} are part of
the COIN-OR project, they are different software.
The \pkg{lpsymphony} package provides an interface to the \emph{SYMPHONY}
software, and is distributed through
\href{https://doi.org/doi:10.18129/B9.bioc.lpsymphony}{Bioconductor}.
We recommend using this solver if the \emph{CBC} and \emph{HiGHS} solvers cannot
be installed.
This solver can use parallel processing to solve problems, so it is
faster than \pkg{Rsymphony} package interface (see below).
}

\item{\code{\link[=add_rsymphony_solver]{add_rsymphony_solver()}}}{
This solver provides an alternative interface to the
\href{https://github.com/coin-or/SYMPHONY}{\emph{SYMPHONY}} solver using
the \pkg{Rsymphony} package.
It is not recommended to use this solver because it has the slowest
performance.
}

}
}
\examples{
\dontrun{
# load data
sim_pu_raster <- get_sim_pu_raster()
sim_features <- get_sim_features()

# create basic problem
p <-
  problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_proportion_decisions()

# create vector to store plot names
n <- c()

# create empty list to store solutions
s <- c()

# if gurobi is installed: create problem with added gurobi solver
if (require("gurobi")) {
  p1 <- p \%>\% add_gurobi_solver(verbose = FALSE)
  n <- c(n, "gurobi")
  s <- c(s, solve(p1))
}

# if cplexAPI is installed: create problem with added CPLEX solver
if (require("cplexAPI")) {
  p2 <- p \%>\% add_cplex_solver(verbose = FALSE)
  n <- c(n, "CPLEX")
  s <- c(s, solve(p2))
}

# if rcbc is installed: create problem with added CBC solver
if (require("rcbc")) {
  p3 <- p \%>\% add_cbc_solver(verbose = FALSE)
  n <- c(n, "CBC")
  s <- c(s, solve(p3))
}

# if highs is installed: create problem with added HiGHs solver
if (require("highs")) {
  p4 <- p \%>\% add_highs_solver(verbose = FALSE)
  n <- c(n, "HiGHS")
  s <- c(s, solve(p4))
}

# create problem with added rsymphony solver
if (require("Rsymphony")) {
  p5 <- p \%>\% add_rsymphony_solver(verbose = FALSE)
  n <- c(n, "Rsymphony")
  s <- c(s, solve(p5))
}

# if lpsymphony is installed: create problem with added lpsymphony solver
if (require("lpsymphony")) {
  p6 <- p \%>\% add_lpsymphony_solver(verbose = FALSE)
  n <- c(n, "lpsymphony")
  s <- c(s, solve(p6))
}

# plot solutions
names(s) <- n
plot(terra::rast(s), axes = FALSE)
}

}
\references{
Hanson JO, Schuster R, Strimas-Mackey M, Morrell N, Edwards BPM, Arcese P,
Bennett JR, and Possingham HP (2025). Systematic conservation prioritization
with the prioritizr R package. \emph{Conservation Biology}, 39: e14376.

Schuster R, Hanson JO, Strimas-Mackey M, and Bennett JR (2020). Exact
integer linear programming solvers outperform simulated annealing for
solving conservation planning problems. \emph{PeerJ}, 8: e9258.
}
\seealso{
Other overviews: 
\code{\link{constraints}},
\code{\link{decisions}},
\code{\link{importance}},
\code{\link{objectives}},
\code{\link{penalties}},
\code{\link{portfolios}},
\code{\link{summaries}},
\code{\link{targets}}
}
\concept{overviews}
