% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/llm_backends.R
\name{submit_llm_pairs}
\alias{submit_llm_pairs}
\title{Backend-agnostic live comparisons for a tibble of pairs}
\usage{
submit_llm_pairs(
  pairs,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  backend = c("openai", "anthropic", "gemini", "together", "ollama"),
  endpoint = c("chat.completions", "responses"),
  api_key = NULL,
  verbose = TRUE,
  status_every = 1,
  progress = TRUE,
  include_raw = FALSE,
  ...
)
}
\arguments{
\item{pairs}{Tibble or data frame with at least columns \code{ID1}, \code{text1},
\code{ID2}, \code{text2}. Typically created by \code{\link[=make_pairs]{make_pairs()}}, \code{\link[=sample_pairs]{sample_pairs()}}, and
\code{\link[=randomize_pair_order]{randomize_pair_order()}}.}

\item{model}{Model identifier for the chosen backend. For \code{"openai"} this
should be an OpenAI model name (for example \code{"gpt-4.1"}, \code{"gpt-5.1"}).
For \code{"anthropic"} and \code{"gemini"}, use the corresponding provider model
names (for example \code{"claude-4-5-sonnet"} or
\code{"gemini-3-pro-preview"}). For "together", use Together.ai model identifiers
such as \code{"deepseek-ai/DeepSeek-R1"} or \code{"deepseek-ai/DeepSeek-V3"}. For
\code{"ollama"}, use a local model name known to the Ollama server (for example
\code{"mistral-small3.2:24b"}, \code{"qwen3:32b"}, \code{"gemma3:27b"}).}

\item{trait_name}{Trait name to pass through to the backend-specific
comparison function (for example \code{"Overall Quality"}).}

\item{trait_description}{Full-text trait description passed to the backend.}

\item{prompt_template}{Prompt template string, typically from
\code{\link[=set_prompt_template]{set_prompt_template()}}.}

\item{backend}{Character scalar indicating which LLM provider to use.
One of \code{"openai"}, \code{"anthropic"}, \code{"gemini"}, \code{"together"}, or \code{"ollama"}.}

\item{endpoint}{Character scalar specifying which endpoint family to use for
backends that support multiple live APIs. For the \code{"openai"} backend this
must be one of \code{"chat.completions"} or \code{"responses"}, matching
\code{\link[=submit_openai_pairs_live]{submit_openai_pairs_live()}}. For \code{"anthropic"}, \code{"gemini"}, \code{"together"},
and \code{"ollama"}, this is currently ignored.}

\item{api_key}{Optional API key for the selected backend. If \code{NULL}, the
backend-specific helper will use its own default environment variable.
For \code{"ollama"}, this argument is ignored (no API key is required for
local inference).}

\item{verbose}{Logical; if \code{TRUE}, prints status, timing, and result
summaries (for backends that support it).}

\item{status_every}{Integer; print status and timing for every
\code{status_every}-th pair. Defaults to 1 (every pair). Errors are always
printed.}

\item{progress}{Logical; if \code{TRUE}, shows a textual progress bar for
backends that support it.}

\item{include_raw}{Logical; if \code{TRUE}, each row of the returned tibble will
include a \code{raw_response} list-column with the parsed JSON body from the
backend (for backends that support this).}

\item{...}{Additional backend-specific parameters. For \code{"openai"} these
are forwarded to \code{\link[=submit_openai_pairs_live]{submit_openai_pairs_live()}} (and ultimately
\code{\link[=openai_compare_pair_live]{openai_compare_pair_live()}}) and typically include \code{temperature},
\code{top_p}, \code{logprobs}, \code{reasoning}, and \code{include_thoughts}. For
\code{"anthropic"} and \code{"gemini"}, they are forwarded to
\code{\link[=submit_anthropic_pairs_live]{submit_anthropic_pairs_live()}} or \code{\link[=submit_gemini_pairs_live]{submit_gemini_pairs_live()}} and
may include options such as \code{max_output_tokens}, \code{include_thoughts}, and
provider-specific controls. For \code{"ollama"}, arguments are forwarded to
\code{\link[=submit_ollama_pairs_live]{submit_ollama_pairs_live()}} and may include \code{host}, \code{think},
\code{num_ctx}, and other Ollama-specific options.}
}
\value{
A tibble with one row per pair and the same columns as the
underlying backend-specific helper for the selected backend. All
backends are intended to return a compatible structure suitable for
\code{\link[=build_bt_data]{build_bt_data()}} and \code{\link[=fit_bt_model]{fit_bt_model()}}.
}
\description{
\code{submit_llm_pairs()} is a backend-neutral wrapper around row-wise comparison
for multiple pairs. It takes a tibble of pairs (\code{ID1}, \code{text1}, \code{ID2},
\code{text2}), submits each pair to the selected backend, and binds the results
into a single tibble.
}
\details{
At present, the following backends are implemented:
\itemize{
\item \code{"openai"}   → \code{\link[=submit_openai_pairs_live]{submit_openai_pairs_live()}}
\item \code{"anthropic"} → \code{\link[=submit_anthropic_pairs_live]{submit_anthropic_pairs_live()}}
\item \code{"gemini"}   → \code{\link[=submit_gemini_pairs_live]{submit_gemini_pairs_live()}}
\item \code{"together"}  → \code{\link[=together_compare_pair_live]{together_compare_pair_live()}}
\item \code{"ollama"}   → \code{\link[=submit_ollama_pairs_live]{submit_ollama_pairs_live()}}
}

Each backend-specific helper returns a tibble with one row per pair and a
compatible set of columns, including a \code{thoughts} column (reasoning /
thinking text when available), \code{content} (visible assistant output),
\code{better_sample}, \code{better_id}, and token usage fields.
}
\examples{
\dontrun{
# Requires an API key for the chosen cloud backend. For OpenAI, set
# OPENAI_API_KEY in your environment. Running these examples will incur
# API usage costs.
#
# For local Ollama use, an Ollama server must be running and the models
# must be pulled in advance. No API key is required for the `"ollama"`
# backend.

data("example_writing_samples", package = "pairwiseLLM")

pairs <- example_writing_samples |>
  make_pairs() |>
  sample_pairs(n_pairs = 5, seed = 123) |>
  randomize_pair_order(seed = 456)

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# Live comparisons for multiple pairs using the OpenAI backend
res_live <- submit_llm_pairs(
  pairs             = pairs,
  model             = "gpt-4.1",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  backend           = "openai",
  endpoint          = "chat.completions",
  temperature       = 0,
  verbose           = TRUE,
  status_every      = 2,
  progress          = TRUE,
  include_raw       = FALSE
)

res_live$better_id

# Live comparisons using a local Ollama backend

res_ollama <- submit_llm_pairs(
  pairs             = pairs,
  model             = "mistral-small3.2:24b",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  backend           = "ollama",
  verbose           = TRUE,
  status_every      = 2,
  progress          = TRUE,
  include_raw       = FALSE,
  think             = FALSE,
  num_ctx           = 8192
)

res_ollama$better_id
}

}
\seealso{
\itemize{
\item \code{\link[=submit_openai_pairs_live]{submit_openai_pairs_live()}}, \code{\link[=submit_anthropic_pairs_live]{submit_anthropic_pairs_live()}},
\code{\link[=submit_gemini_pairs_live]{submit_gemini_pairs_live()}}, \code{\link[=submit_together_pairs_live]{submit_together_pairs_live()}}, and
\code{\link[=submit_ollama_pairs_live]{submit_ollama_pairs_live()}} for backend-specific implementations.
\item \code{\link[=llm_compare_pair]{llm_compare_pair()}} for single-pair comparisons.
\item \code{\link[=build_bt_data]{build_bt_data()}} and \code{\link[=fit_bt_model]{fit_bt_model()}} for Bradley–Terry modelling of
comparison results.
}
}
