#' Spinner-Based Geometric Multivariate Outlier Detection
#'
#' Computes multivariate outlier scores using random directional probing
#' of standardized observations. The method captures both radial extremeness
#' and angular alignment by projecting observations onto multiple random
#' directions ("spins") and aggregating projection-based deviations.
#'
#' In addition to a global outlier score, the function provides
#' dimension-level contribution measures that attribute each observation's
#' outlyingness to the original variables, enabling post hoc interpretability.
#'
#' Robust centering, scaling, and optional covariance adjustment ensure
#' affine invariance and resistance to marginal contamination.
#'
#' @param X A numeric matrix or data frame of dimension \eqn{n \times p},
#'   with rows as observations and columns as variables.
#' @param n_spins Integer specifying the number of random directions for
#'   directional probing. Larger values increase stability at higher
#'   computational cost.
#' @param robust Logical; if \code{TRUE}, robust centering (median) and
#'   scaling (MAD) are applied. If \code{FALSE}, classical centering and
#'   scaling are used.
#' @param cov_adjust Logical; if \code{TRUE}, observations are whitened using
#'   a robust or classical covariance estimate (depending on \code{robust})
#'   to remove linear dependence.
#' @param contrib_quantile Numeric in \eqn{(0,1)} specifying the quantile
#'   threshold to select influential spinner directions for computing
#'   dimension-level contributions (default 0.90).
#' @param plot_top_n Optional integer; if provided and
#'   \code{make_contrib_plot = TRUE}, a stacked bar plot of dimension
#'   contributions is produced for the top \code{plot_top_n} observations.
#' @param make_contrib_plot Logical; if \code{TRUE}, generates a stacked bar
#'   plot summarizing dimension-level contributions for the most outlying
#'   observations.
#' @param seed Optional integer seed for reproducibility of random spinner
#'   directions.
#'
#' @return A list with the following components:
#' \describe{
#'   \item{score}{Numeric vector of length \eqn{n} with Spinner outlier scores.}
#'   \item{score_align}{Numeric vector capturing angular alignment per observation.}
#'   \item{mean_proj}{Mean absolute projection per observation across all spinner directions.}
#'   \item{proj_matrix}{Matrix of signed projections of observations onto spinner directions.}
#'   \item{dim_contrib_raw}{Matrix of raw dimension-level contributions per observation.}
#'   \item{dim_contrib_norm}{Row-normalized contributions, interpretable as relative attribution weights.}
#'   \item{top_spins}{List of influential spinner directions selected for each observation.}
#'   \item{contrib_plot}{A \code{ggplot2} object showing contributions for the most extreme observations, or \code{NULL}.}
#'   \item{n_spins}{Number of random directions used.}
#'   \item{robust_center}{Vector of location estimates used for centering.}
#'   \item{cov_adjust}{Logical indicating whether covariance adjustment was applied.}
#' }
#'
#' @details
#' The Spinner score combines two complementary components:
#' \enumerate{
#'   \item Radial deviation, measured as squared deviations of projections
#'   from their marginal centers.
#'   \item Angular alignment, capturing whether an observation consistently
#'   aligns with specific directions in high-dimensional space.
#' }
#' Dimension-level contributions are computed by backprojecting influential
#' spinner directions to the original coordinate system. Only directions
#' with large projection magnitudes are retained, preserving rotational
#' invariance while enabling interpretability.
#'
#' The method is motivated by ongoing research. A detailed theoretical
#' treatment and empirical evaluation are provided in a manuscript
#' currently under review.
#'
#' @references
#' Economou, P. (2026). \emph{Spinner: A Geometric Multivariate Outlier
#' Detection Method Using Random Directional Probing}. Manuscript under review.
#'
#' @examples
#' set.seed(123)
#' X <- matrix(rnorm(40), ncol = 4)
#' res <- spinner_outlier_score(
#'   X,
#'   n_spins = 100,
#'   robust = TRUE,
#'   cov_adjust = TRUE,
#'   contrib_quantile = 0.9,
#'   plot_top_n = 3,
#'   make_contrib_plot = TRUE
#' )
#' head(res$score)
#' if (!is.null(res$contrib_plot)) print(res$contrib_plot)
#'
#' @importFrom stats median mad cov rnorm quantile
#' @importFrom MASS cov.rob
#' @import ggplot2
#' @import tidyr
#' @export


spinner_outlier_score <- function(X,
                                  n_spins = 1000,
                                  robust = TRUE,
                                  cov_adjust = TRUE,
                                  contrib_quantile = 0.90,
                                  plot_top_n = NULL,
                                  make_contrib_plot = FALSE,
                                  seed = NULL) {

  stopifnot(is.matrix(X) || is.data.frame(X))
  stopifnot(n_spins > 0, n_spins == as.integer(n_spins))
  stopifnot(contrib_quantile > 0 && contrib_quantile < 1)
  if (anyNA(X)) stop("X contains NA values. Please handle missing data before calling spinner_outlier_score().")
  if (!is.null(seed)) set.seed(seed)

  X <- as.matrix(X)
  n <- nrow(X)
  p <- ncol(X)

  # Observation labels (prefer row names if available)
  obs_labels <- rownames(X)
  if (is.null(obs_labels)) {
    obs_labels <- paste0("Obs_", seq_len(n))
  }

  # Robust centering and scaling
  if (robust) {
    mu <- apply(X, 2, stats::median)
    s  <- apply(X, 2, stats::mad)
    s[s == 0] <- 1
    Xs <- sweep(sweep(X, 2, mu), 2, s, "/")
  } else {
    Xs <- scale(X)
    mu <- rep(0, p)
  }

  # Covariance adjustment
  if (cov_adjust) {
    if (robust) {
      S <- MASS::cov.rob(Xs)$cov
    } else {
      S <- stats::cov(Xs)
    }
    eig <- eigen(S, symmetric = TRUE)
    S_inv_sqrt <- eig$vectors %*%
      diag(1 / sqrt(pmax(eig$values, .Machine$double.eps))) %*%
      t(eig$vectors)
    Xs <- Xs %*% S_inv_sqrt
  }

  # Directions and radii
  radii <- sqrt(rowSums(Xs^2))
  radii[radii == 0] <- .Machine$double.eps
  U <- Xs / radii

  # Spinner directions
  V <- matrix(stats::rnorm(n_spins * p), ncol = p)
  V <- V / sqrt(rowSums(V^2))

  # Projections
  proj_matrix <- Xs %*% t(V)
  abs_proj    <- abs(proj_matrix)
  mean_proj   <- rowMeans(abs_proj)
  score_align <- 1 - mean_proj / max(abs_proj)
  proj_dev <- sweep(proj_matrix, 2, apply(proj_matrix, 2, stats::median), "-")
  score_proj <- rowMeans(proj_dev^2) * score_align^2

  # Dimension-level contributions
  dim_contrib_raw <- matrix(0, nrow = n, ncol = p)
  top_spins <- vector("list", n)
  for (i in seq_len(n)) {
    Pi  <- abs_proj[i, ]
    thr <- stats::quantile(Pi, contrib_quantile)
    Ki  <- which(Pi >= thr)
    top_spins[[i]] <- Ki
    zi_abs <- abs(Xs[i, ])
    for (k in Ki) {
      dim_contrib_raw[i, ] <- dim_contrib_raw[i, ] + abs(V[k, ]) * zi_abs * Pi[k]
    }
  }
  dim_contrib_norm <- dim_contrib_raw / rowSums(dim_contrib_raw)
  colnames(dim_contrib_raw)  <- colnames(X)
  colnames(dim_contrib_norm) <- colnames(X)

  # Optional contribution plot
  contrib_plot <- NULL
  if (make_contrib_plot && !is.null(plot_top_n)) {

    # 1. Identify indices of top scoring observations
    idx <- order(score_proj, decreasing = TRUE)[1:plot_top_n]

    # 2. Extract normalized dimension contributions
    df_bar <- as.data.frame(dim_contrib_norm[idx, , drop = FALSE])

    # 3. Add observation labels (row names if available)
    df_bar$obs_label <- factor(
      obs_labels[idx],
      levels = obs_labels[idx]
    )

    # 4. Pivot longer for ggplot
    df_long <- tidyr::pivot_longer(
      df_bar,
      cols = -obs_label,
      names_to = "dimension",
      values_to = "contribution"
    )

    # 5. Create contribution bar plot
    contrib_plot <- ggplot2::ggplot(
      df_long,
      ggplot2::aes(
        x = obs_label,
        y = contribution,
        fill = dimension
      )
    ) +
      ggplot2::geom_col(width = 0.85) +
      ggplot2::labs(
        title = "Dimension Contributions to Spinner Score",
        subtitle = "Top outlying observations",
        x = "Observation",
        y = "Normalized contribution"
      ) +
      ggplot2::scale_y_continuous(expand = c(0, 0)) +
      ggplot2::theme_minimal(base_size = 14) +
      ggplot2::theme(
        axis.text.x = ggplot2::element_text(angle = 45, hjust = 1),
        legend.position = "right"
      )
  }

  # Output
  list(
    score            = score_proj,
    score_align      = score_align,
    mean_proj        = mean_proj,
    proj_matrix      = proj_matrix,
    dim_contrib_raw  = dim_contrib_raw,
    dim_contrib_norm = dim_contrib_norm,
    top_spins        = top_spins,
    contrib_plot     = contrib_plot,
    n_spins          = n_spins,
    robust_center    = mu,
    cov_adjust       = cov_adjust
  )
}


utils::globalVariables(c("obs_label", "dimension", "contribution"))
