% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mland_metrics.R
\name{mland_metrics}
\alias{mland_metrics}
\title{Calculates landscape metrics}
\usage{
mland_metrics(
  x,
  raster = NULL,
  points = NULL,
  radii = NULL,
  classes = NULL,
  level = NULL,
  metric = NULL,
  name = NULL,
  type = NULL,
  what = NULL,
  report_absences = TRUE,
  absence_values = NULL,
  ext_calc = NULL,
  na.exclude = TRUE,
  coords = FALSE,
  update = NULL,
  output = "MLM",
  progress = TRUE,
  ...
)
}
\arguments{
\item{x}{An object of class 'Multiland' generated with \code{\link[=mland]{mland()}}.}

\item{raster}{Vector depicting the raster layers of \code{x} from which metrics will be
calculated. It can be a numeric vector, for rasterlayer numbers, or a character vector, for
raster layer names (if provided during the generation of \code{x}). If NULL, all raster layers will be
considered.}

\item{points}{Numeric or character vector with the points from which metrics will be calculated.
If NULL, all points will be considered. See Details.}

\item{radii}{Numeric vector depicting the radii from which metrics will be calculated.
If NULL, all radii will be considered.}

\item{classes}{List containing the classes or classes names from which metrics will be calculated.
If NULL, all classes will be considered. See Details.}

\item{level, metric, name, type, what}{Arguments passed to \code{\link[landscapemetrics:calculate_lsm]{landscapemetrics::calculate_lsm()}},
which define which metrics will be calculated. See Details.}

\item{report_absences}{Logical. If TRUE (default), intersections with absences of particular
classes will be returned in the final data.frame. See Details.}

\item{absence_values}{A list depicting which value for each class-level metric should be printed
if \code{report_absences = TRUE}. See Details.}

\item{ext_calc}{A list containing vectors, each one of length equal to 2 or more: the first element of the vector
with the identification number of the extra raster layer defined in \code{x}, and next elements with a string with
the name of the function to be applied to the defined raster. See Details.}

\item{na.exclude}{Logical. Whether to exclude (default) or not the NA values when performing
extra calculations to extra raster layers. Only applies if \code{ext_calc} is not NULL. See Details.}

\item{coords}{Logical. If TRUE, the coordinates of the points will be returned in the data.frame
containing the values of the required metrics. Default FALSE.}

\item{update}{An object of class 'MultiLandMetrics', if it is intended to be updated with
new or updated metrics data. See Details.}

\item{output}{Either "MLM" (default) to output an object of class 'MultiLandMetrics' or "data" to output
only the data.frame with metric values.}

\item{progress}{Logical. If TRUE (default), progress of calculations will be printed.}

\item{...}{Other arguments passed to \code{\link[landscapemetrics:calculate_lsm]{landscapemetrics::calculate_lsm()}}. See Details.}
}
\value{
If \code{output = "MLM"}, an object of class 'MultiLandMetrics' will be returned. This object
can then be passed to functions \code{\link[=metrics_corr]{metrics_corr()}}, \link{metrics_plots}, \code{\link[=metrics_filter]{metrics_filter()}}, \code{\link[=metrics_gradient]{metrics_gradient()}}
and \code{\link[=metrics_bind]{metrics_bind()}}. See
?MultiLandMetrics for more information regarding the content of this object. Otherwise,
if \code{output = "data"}, only a data.frame with the calculated metrics will be returned.
}
\description{
Calculates landscape metrics of patch, class and/or landscape level via the package
\code{landscapemetrics} and user-defined functions from an object of class 'MultiLand'.
}
\details{
Calculates landscape metrics from an object of class \code{MultiLand}
created with \code{\link[=mland]{mland()}}. The function allows to define which metrics will be calculated in the
form defined by the function \code{\link[landscapemetrics:calculate_lsm]{landscapemetrics::calculate_lsm()}} from package \code{landscapemetrics}, by
specifying one or more of the following arguments:
\itemize{
\item level: level of metrics. Either "patch", "class" or "landscape" (or vector with combination).
\item metric: abbreviation of metrics (e.g. "area").
\item name: full name of metrics (e.g. "core area").
\item type: type according to FRAGSTATS grouping (e.g. "aggregation metrics").
\item what: selected level of metrics: either "patch", "class" or "landscape". It is also possible to
specify functions as a vector of strings, e.g. what = c("lsm_c_ca", "lsm_l_ta").
}

Available metrics can be seen in \code{\link[=metrics_list]{metrics_list()}} and in the associated documentation of package
\code{landscapemetrics}.

\code{\link[=mland_metrics]{mland_metrics()}} also allows to define some other parameters that filter how metrics are calculated,
by defining the raster layers, points, radii and classes to be taken into account.

If \code{report_absences = TRUE} (default), the function will print values of class-level metrics
from classes that are not present in particular landscapes, as a distinct row in the final
data.frame. This is particularly useful for certain class-level metrics in which the absence
of the class should be acknowledged, for instance, the percentage of landscape ('pland') for a forest class. For this
metric, a value of 0 (zero) should be printed for those landscapes where the class forest is not present.
By default, if \code{report_absences = TRUE}, the function will consider \code{NA} as the value
to be declared in the case that the class is absent in the landscape. To declare a different value
for a particular class-level metric, this can be declared inside argument \code{absence_values}. If
not NULL, this must be a list with the value that one ore more class-level metric should
have in the case of an absence of a class. For example, in the case of "pland", the argument
should be defined as follows: \code{absence_values = list("pland" = 0)}. Note that the metric must be
identified with its abbreviation. You can see abbreviations for all available metrics in \code{\link[=metrics_list]{metrics_list()}},
under the column "metric".

If argument \code{points} is a character vector,
\code{\link[=mland_metrics]{mland_metrics()}} assumes that the 'MultiLand' object inputted in argument \code{x} was created with
\code{site_ref = TRUE}. This is, there is an column/attribute in the points layer with the names for
each distinct point. Therefore, the inputted values in argument \code{points} will be taken as these
identification names. Otherwise, if a numeric vector is inputted, these values
will be taken as the automatically generated point ids (created when running \code{\link[=mland]{mland()}}).

The user may specify which classes will be considered when calculating the metrics, by
passing this information in the argument \code{classes}. Of course, this information only applies for
class-level metrics. The argument must be a list with as many elements as raster layers to be
considered (defined in argument \code{raster}, in ascending order: 1, 2, 3, ...). Each element must be a numeric vector
with the classes values (raster values) to be considered, or a character vector with
the names of the classes (if provided when generating \code{x}).

Other arguments can be passed to function \code{\link[landscapemetrics:calculate_lsm]{landscapemetrics::calculate_lsm()}} through argument
\code{...}. These include specific arguments relative to the calculation of particular landscape
metrics. See the documentation of this function for more information.

Extra calculations can be performed through \code{ext_calc}. The functions defined here will take
the values of the extra raster layers defined in \code{x} as input. For instance, a plausible definition
could be \code{ext_calc = list(1, "mean")}, which will take the values from the extra raster layer 1,
and calculate its mean for each landscape. If \code{na.exclude = TRUE} (default), NA values will be excluded
from this task.

A previously generated 'MultiLandMetrics' object can be updated with new information regarding
other metrics, probably from other points, radii, raster layers, etc, that haven´t been
calculated in the previous time (or not). In this way, the returned object will be
the object provided in this argument, plus the additions of information about new metrics, and
changes to previously metric calculations. Note that if a particular metric is calculated for a
given raster layer, points, radii and or class, that were previously generated in the object
provided in \code{update}, the information of these metrics from the latter will be overwritten. Also
note that if in the previous 'MultiLandMetrics' object \code{report_absences} was \code{TRUE} for a given
set of metrics and other parameters (e.g. points, radii, raster layers, etc.), and in the new call
\code{report_absences} is \code{FALSE} (for the same set of other parameters), the rows depicting landscapes
with empty classes from the previous call will be mantained. If the intention is the removal of these rows,
the user should create a fresh new 'MultiLandMetrics' from scratch.
}
\examples{
\donttest{
# Loads a 'MultiLand' object
ernesdesign <- system.file("extdata", "ernesdesign.zip", package = "multilandr")
ernesdesign <- mland_load(ernesdesign)

# Creates a 'MultiLandMetrics' object. It will calculate the "percentage of landscape"
# ("pland") and "edge density" ("ed") for all classes. Note that an absence value
# for each metric is declared, as the absence of a class for these metrics should be
# acknowledged as a 0 (percentage of zero and zero patches).
ed_metrics <- mland_metrics(ernesdesign, level = "class", metric = c("pland", "ed"),
                            absence_values = list("pland" = 0))

# Returns data.frame with the values of all metrics for each landscape
head(ed_metrics@data)

# Shows which metrics were calculated and are contained in the data.frame
ed_metrics@metrics

# If output = "data", only the data.frame will be returned
data <- mland_metrics(ernesdesign, level = "class", metric = "pland",
                      classes = c("Forest", "Crops"),
                      absence_values = list("pland" = 0),
                      output = "data")

# Calculate landscape metrics plus extra calculations for extra rasterlayer 1,
# the mean value, and a user defined function, which is the mean divided
# standard deviation.

# User-defined function
mean_sd <- function(x){ mean(x)/sd(x) }

ed_metrics2 <- mland_metrics(ernesdesign, level = "class",
                             metric = c("pland", "ed"),
                             absence_values = list("pland" = 0),
                             ext_calc = list(c(1, "mean"), c(1, "mean_sd")))

# We can calculate metrics for extra raster layers only
ed_metrics3 <- mland_metrics(ernesdesign, ext_calc = list(c(1, "mean", "mean_sd")))

# If metrics of different levels must be calculated, a better approach is to declare
# them inside the argument 'what', by naming the function associated with the metric.
# Also in this case, only the landscapes with a radius of 5000 m are considered.
# A list of available metrics with its names, abbreviations and function names can
# be seen in metrics_list() and in the documentation of the package landscapemetrics.
ed_metrics4 <- mland_metrics(ernesdesign,
                             what = c("lsm_c_area_mn", "lsm_l_ed", "lsm_l_shdi"),
                             radii = 5000)

# Calculates patch-level metrics of a particular landscape
ed_patchs <- mland_metrics(ernesdesign, points = "Algarrobo",
                           level = "patch", class = "Forest",
                           radii = 1000)
}
}
\seealso{
\code{\link[=metrics_corr]{metrics_corr()}}, \code{\link[=metrics_plots]{metrics_plots()}}, \code{\link[=metrics_filter]{metrics_filter()}}, \code{\link[=metrics_gradient]{metrics_gradient()}}, \code{\link[=metrics_bind]{metrics_bind()}}
}
