% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generate_points.R
\name{generate_points}
\alias{generate_points}
\title{Generates point coordinates}
\usage{
generate_points(
  raster,
  approach = "grid",
  n = NULL,
  padding = 0,
  try = NULL,
  values = NULL,
  patch_conditions = NULL,
  trim = TRUE,
  attempts = 10,
  distance = NULL,
  offset = FALSE,
  closest_cell = FALSE,
  parallel = FALSE,
  cores = 1,
  progress = TRUE
)
}
\arguments{
\item{raster}{An object of class 'SpatRaster', 'RasterLayer', 'RasterStack' or 'RasterBrick'.}

\item{approach}{One of the following: "grid" to generate points through a grid, "random" to
generate points at random locations, or "patch" to generate points inside patches that meet
pre-defined conditions. See Details.}

\item{n}{Number of point to generate.}

\item{padding}{Numeric. Width (in meters) of the internal margin around the raster that will be
discarded from the analysis. See Details.}

\item{try}{Number of points to be generated in
each turn. Only applies if \code{approach = "random"}. See Details.}

\item{values}{The values of the rasterlayer where points should be placed. Only applies if
\code{approach = "random"}.}

\item{patch_conditions}{The conditions that patches must meet to be included as the
patches from which points will be generated. Only applies if \code{approach = "patch"}. See Details.}

\item{trim}{Logical. If TRUE (default) the number of final points will be trimmed to the value
defined in \code{n}.}

\item{attempts}{Number of attempts to generate new random points given the total required
points (\code{n}) and the minimum distance required in \code{distance}. Only applies if
\code{approach = "random"}. See Details.}

\item{distance}{Distance between points of the grid (if \code{approach = "grid"}) or minimum distance
between generated points (if \code{approach = "random"}).}

\item{offset}{Logical. If TRUE, each point coordinates will be randomly displaced around the area
occupied by the raster cell size. If FALSE (default), each point will be located at the center of
a given raster cell. Only applies if \code{approach = "random"}.}

\item{closest_cell}{Logical. If \code{approach = "patch"}, whether to return the coordinates of each patch centroid even if
they fall outside the patch (FALSE, default) or to move the point to the
closest cell of the patch if this happens (TRUE).}

\item{parallel}{Logical. If TRUE, part of the processes will be parallelized. See Details.}

\item{cores}{Number of cores to use if \code{parallel = TRUE}.}

\item{progress}{Logical. If TRUE (default), progress of the analysis will be printed.}
}
\value{
An object of class 'SpatVector' containing the coordinates of the generated points.
}
\description{
Generates point coordinates over a rasterlayer extent.
}
\details{
If \code{approach = "random"}, the user can restrict the locations of new generated points
inside raster cells with certain value or values, by defining them in \code{values}. Also a minimum distance
between the generated points can be defined in \code{distance} (also applies for the resolution of
the grid if \code{approach = "grid"}).

If \code{approach = "random"} and a minimum distance was defined, the function will generate new
"random" points in sequential passes. In each pass, the function will try to generate new points
taking into account the minimum distance between points, by randomly generating a number of points as
defined in \code{try}. The function will perform this task until the new generated points is equal or
higher than \code{n}. If \code{try = NULL} (default), \code{try} will equals \code{n}. If in each turn no new points
were added (i.e. new points were located at less than the minimum distance to other previously
generated points), the function will record this event. If this event happens more than the
number of times defined in \code{attempts} before the total generated points equals \code{n}, the function will
terminate, and return the points that were successfully generated given the required parameters.
The user may try different values for \code{n}, \code{try} and \code{attempts} to get a desirable result.

If \code{approach = "patch"}, the function will return as many points as patches that meet certain
conditions in relation to pre-defined metric values. Conditions can be defined in
argument \code{patch_conditions}, for which the helper function \code{\link[=conditions]{conditions()}} is available:

\preformatted{
 conditions(list(class, metric, minimum value, maximum value),
            list(class, metric, minimum value, maximum value), ...)
}
\itemize{
\item class: the class (raster value) of the patch that must meet the defined conditions. More than one class can
be specified.
\item metric: the patch-level metric whose values must meet the defined conditions. Only one metric
per condition can be defined. Available patch-level metrics can be found in \code{\link[=metrics_list]{metrics_list()}} and in
documentation of the package \code{landscapemetrics}.
\item minimum value: the minimum value that the metric must have for the retained patches. If equal
to -Inf, and a maximum value is defined, patches whose values in the defined metric are equal
or lower to the maximum value will be retained.
\item maximum value: the maximum value that the metric must have in the retained patches. If equal
to Inf, and a minimum value is defined, patches whose values in the defined metric are equal
or higher to the minimum value will be retained.
}

Retained patches will be those patches that meet all patch conditions at the same time. Returned
point's coordinates will equal the centroid of each patch. If \code{closest_cell = TRUE}, the point's coordinates of the
centroids that did not fall inside the patch will be moved to the closest cell belonging to that
patch.

To avoid generating points to close to the boundaries of the raster, the outer borders of the
raster can be discarded from the analysis, by considering the width inputted in \code{padding}.

If \code{parallel = TRUE} the function will parallelize part of the processes. Parallelization
is done to obtain the coordinates of the patches if \code{approach = "patch"}. The number of
cores must be declared in \code{cores} (parallelization requires at least two cores). To use this
functionality, package \code{parallel} must be installed. So far, parallelization will run
in LINUX and MAC, but not in Windows.
}
\examples{
\donttest{
# Loads raster
elchaco <- terra::rast(system.file("extdata", "elchaco.tif", package = "multilandr"))

# Returns points at "random" locations, but inside cells of value equals to 1.
chaco_coords <- generate_points(elchaco, approach = "random", values = 1, n = 500)

# The same but points must be separated by at least 300 m between each other. Also, each point
# is randomly displaced inside the raster cell.
chaco_coords2 <- generate_points(elchaco, approach = "random", values = 1, n = 500,
                                 try = 100, distance = 300, offset = TRUE)

# Returns as many points as patches that meet the defined condition. This is
# all patches of value equal to 1 of area between 9 and 11 hectares.
patch_sites <- generate_points(elchaco, approach = "patch",
                               patch_conditions = conditions(list(1, "area", 8, 12)),
                               padding = 2000)
}
}
\seealso{
\code{\link[=mland]{mland()}}
}
