% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimation.R
\name{mtar}
\alias{mtar}
\title{Bayesian estimation of a multivariate Threshold Autoregressive (TAR) model.}
\usage{
mtar(
  formula,
  data,
  subset,
  Intercept = TRUE,
  trend = c("none", "linear", "quadratic"),
  nseason = NULL,
  ars = ars(),
  row.names,
  dist = c("Gaussian", "Student-t", "Hyperbolic", "Laplace", "Slash",
    "Contaminated normal", "Skew-Student-t", "Skew-normal"),
  prior = list(),
  n.sim = 500,
  n.burnin = 100,
  n.thin = 1,
  ssvs = FALSE,
  setar = NULL,
  progress = TRUE,
  ...
)
}
\arguments{
\item{formula}{A three-part expression of class \code{Formula} describing the TAR model to be fitted.
The first part specifies the variables in the multivariate output series, the second part
defines the threshold series, and the third part specifies the variables in the multivariate
exogenous series.}

\item{data}{A data frame containing the variables in the model. If not found in \code{data}, the
variables are taken from \code{environment(formula)}, typically the environment from
which \code{mtar_grid()} is called.}

\item{subset}{An optional vector specifying a subset of observations to be used in the fitting process.}

\item{Intercept}{An optional logical indicating whether an intercept should be included within each regime.}

\item{trend}{An optional character string specifying the degree of deterministic time trend to be
included in each regime. Available options are \code{"linear"}, \code{"quadratic"}, and
\code{"none"}. By default, \code{trend} is set to \code{"none"}.}

\item{nseason}{An optional integer, greater than or equal to 2, specifying the number of seasonal periods.
When provided, \code{nseason - 1} seasonal dummy variables are added to the regressors within each regime.
By default, \code{nseason} is set to \code{NULL}, thereby indicating that the TAR model has no seasonal effects.}

\item{ars}{A list defining the autoregressive structure of the model. It contains four
components: the number of regimes (\code{nregim}), the autoregressive order within each
regime (\code{p}), and the maximum lags for the exogenous (\code{q}) and threshold
(\code{d}) series in each regime. The object can be validated using the helper
function \code{ars()}.}

\item{row.names}{An optional variable in \code{data} labelling the time points corresponding to each row of the data set.}

\item{dist}{A character string specifying the multivariate distributions used to model the noise
process. Available options are \code{"Gaussian"}, \code{"Student-t"}, \code{"Slash"},
\code{"Hyperbolic"}, \code{"Laplace"}, \code{"Contaminated normal"},
\code{"Skew-normal"}, and \code{"Skew-Student-t"}. By default, \code{dist} is set to
\code{"Gaussian"}.}

\item{prior}{An optional list specifying the hyperparameter values that define the prior
distribution. This list can be validated using the \code{priors()} function. By default,
\code{prior} is set to an empty list, thereby indicating that the hyperparameter values
should be set so that a non-informative prior distribution is obtained.}

\item{n.sim}{An optional positive integer specifying the number of simulation iterations after the
burn-in period. By default, \code{n.sim} is set to \code{500}.}

\item{n.burnin}{An optional positive integer specifying the number of burn-in iterations. By default,
\code{n.burnin} is set to \code{100}.}

\item{n.thin}{An optional positive integer specifying the thinning interval. By default,
\code{n.thin} is set to \code{1}.}

\item{ssvs}{An optional logical indicating whether the Stochastic Search Variable Selection (SSVS)
procedure should be applied to identify relevant lags of the output, exogenous, and threshold
series. By default, \code{ssvs} is set to \code{FALSE}.}

\item{setar}{An optional positive integer indicating the component of the output series used as the
threshold variable. By default, \code{setar} is set to \code{NULL}, indicating that the
fitted model is not a SETAR model.}

\item{progress}{An optional logical indicating whether a progress bar should be displayed during
execution. By default, \code{progress} is set to \code{TRUE}.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
an object of class \emph{mtar} in which the main results of the model fitted to the data are stored, i.e., a
list with components including
\tabular{ll}{
\code{chains}   \tab list with several arrays, which store the values of each model parameter in each iteration of the simulation,\cr
\tab \cr
\code{n.sim}    \tab number of iterations of the simulation after the burn-in period,\cr
\tab \cr
\code{n.burnin} \tab number of burn-in iterations in the simulation,\cr
\tab \cr
\code{n.thin}   \tab thinning interval in the simulation,\cr
\tab \cr
\code{ars}      \tab list composed of four objects, namely: \code{nregim}, \code{p}, \code{q} and \code{d},
                     each of which corresponds to a vector of non-negative integers with as
                     many elements as there are regimes in the fitted TAR model,\cr
\tab \cr
\code{dist}     \tab name of the multivariate distribution used to describe the behavior of
                     the noise process,\cr
\tab \cr
\code{threshold.series}  \tab vector with the values of the threshold series,\cr
\tab \cr
\code{output.series}   \tab matrix with the values of the output series,\cr
\tab \cr
\code{exogenous.series} \tab matrix with the values of the exogenous series,\cr
\tab \cr
\code{Intercept}    \tab If \code{TRUE}, then the model included an intercept term in each regime,\cr
\tab \cr
\code{trend}    \tab the degree of the deterministic time trend, if any,\cr
\tab \cr
\code{nseason}    \tab the number of seasonal periods, if any,\cr
\tab \cr
\code{formula}      \tab the formula,\cr
\tab \cr
\code{call}         \tab the original function call.\cr
}
}
\description{
This function implements a Gibbs sampling algorithm to draw samples from the
posterior distribution of the parameters of a multivariate Threshold Autoregressive (TAR)
model and its special cases as SETAR and VAR models. The procedure accommodates a wide
range of noise process distributions, including Gaussian, Student-\eqn{t}, Slash, Symmetric
Hyperbolic, Contaminated normal, Laplace, Skew-normal, and Skew-Student-\eqn{t}.
}
\examples{
\donttest{
###### Example 1: Returns of the closing prices of three financial indexes
data(returns)
fit1 <- mtar(~ COLCAP + BOVESPA | SP500, data=returns, row.names=Date,
             subset={Date<="2016-03-14"}, dist="Student-t",
             ars=ars(nregim=3,p=c(1,1,2)), n.burnin=2000, n.sim=3000,
             n.thin=2, ssvs=TRUE)
summary(fit1)

###### Example 2: Rainfall and two river flows in Colombia
data(riverflows)
fit2 <- mtar(~ Bedon + LaPlata | Rainfall, data=riverflows, row.names=Date,
             subset={Date<="2009-04-04"}, dist="Laplace", ssvs=TRUE,
             ars=ars(nregim=3,p=5), n.burnin=2000, n.sim=3000, n.thin=2)
summary(fit2)

###### Example 3: Temperature, precipitation, and two river flows in Iceland
data(iceland.rf)
fit3 <- mtar(~ Jokulsa + Vatnsdalsa | Temperature | Precipitation,
             data=iceland.rf, subset={Date<="1974-12-21"}, row.names=Date,
             ars=ars(nregim=2,p=15,q=4,d=2), n.burnin=2000, n.sim=3000,
             n.thin=2, dist="Slash")
summary(fit3)
}


}
\references{
Nieto, F.H. (2005) Modeling Bivariate Threshold Autoregressive Processes in the Presence of Missing Data.
            Communications in Statistics - Theory and Methods, 34, 905-930.

Romero, L.V. and Calderon, S.A. (2021) Bayesian estimation of a multivariate TAR model when the noise
            process follows a Student-t distribution. Communications in Statistics - Theory and Methods, 50, 2508-2530.

Calderon, S.A. and Nieto, F.H. (2017) Bayesian analysis of multivariate threshold autoregressive models
            with missing data. Communications in Statistics - Theory and Methods, 46, 296-318.

Vanegas, L.H. and Calderón, S.A. and Rondón, L.M. (2025) Bayesian estimation of a multivariate tar model when the
            noise process distribution belongs to the class of gaussian variance mixtures. International Journal
            of Forecasting.
}
\seealso{
\link{DIC}, \link{WAIC}
}
