% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msltrend.R
\name{msl.trend}
\alias{msl.trend}
\title{Isolate trend component from mean sea level records.}
\usage{
msl.trend(file, station_name = " ", fillgaps = 1, iter = 10000,
  plot = TRUE)
}
\arguments{
\item{file}{csv format input file \strong{with no header row} of annual
average water levels. This file must contain 2 columns with the first column
the time period (in years) and the second column annual average ocean water
levels (in millimetres). Missing data must be denoted by \dQuote{NA}. Missing
data and maximum missing data gap are limited to 15\% and 5\%, respectively,
of the data record. The minimum length data record processed by the package
is 80 years.

\strong{Warning: } If input data files do not conform to these pre-conditions,
the analysis will be terminated. It should be further noted that the existence
of quasi 60 year oscillations in global mean sea level have been well recognised
in the literature. Therefore, in order to be effective for climate change and
sea level research, only input files with a minimum length exceeding 80 years
have been considered in order that the package can identify and isloate such
signals.}

\item{station_name}{character string, providing the name of the data record.

\strong{Note: }This field can be left blank, however, it is retained for use
in banner labelling of all plotting and pdf outputs.}

\item{fillgaps}{numeric, provides 3 alternative gap filling procedures for
missing data. The default procedure (fillgaps = 1) is based on iterative gap
filling using Singular Spectrum Analysis (refer \code{\link[Rssa]{igapfill}})
. The alternatives (2 and 3) are based on linear interpolation and cubic
spline interpolation, respectively (refer \code{\link[zoo]{na.approx}}).

\strong{Note: }Gap filled portions of the time series are denoted in red on
the default screen plot. This is done specifically to provide ready visual
observation to discern if the selected gap filling method provides an
appropriate estimate within the gaps in keeping with the remainder of the
historical record. Depending on the nature of the record and extent of gaps,
some trial and error between alternatives might be necessary to optimise gap
filling.}

\item{iter}{numeric, enables a user defined number of iterations for
bootstrapping to determine error margins. The user range is [500 to 10000]
where 10000 is the default setting.

\strong{Warning: }Although the default setting provides a more accurate basis
for estimating error margins, the degree of iterations slows the analysis and
can take several minutes to run.}

\item{plot}{logical, if \sQuote{TRUE} then the original time series is
plotted to the screen along with the trend component and the result of gap
filling (where necessary). 95\% confidence intervals have also been applied.
Default = TRUE.}
}
\value{
An object of class \dQuote{msl.trend} is returned with the following
elements:
\describe{
 \item{\strong{$Station.Name: }}{the name of the data record.}
 \item{\strong{$Summary: }}{a summary data frame of the relevant attributes
 relating to the trend and
  the inputted annual average data set, including:}
 \itemize{
  \item{$Year: input data; }
  \item{$MSL: input data; }
  \item{$Trend: mean sea level trend; }
  \item{$TrendSD: standard deviation of the determined mean sea level
  trend; }
  \item{$Vel: velocity (or first derivative) of mean sea level trend
  (mm/year); }
  \item{$VelSD: standard deviation of the velocity of the mean sea level
  trend; }
  \item{$Acc: acceleration (or second derivative) of mean sea level trend
  (mm/year/year); }
  \item{$AccSD: standard deviation of the acceleration of the mean sea level
  trend; }
  \item{$Resids: time series of uncorrelated residuals; and }
  \item{$FilledTS: gap-filled time series (where necessary). }
   }
}

\describe{
 \item{\strong{$Velocity: }}{outputs of the peak velocity and the year in
 which it occurred.}
 \item{\strong{$Acceleration: }}{outputs of the peak acceleration and the
 year in which it occurred.}
 \item{\strong{$Record.Length: }}{outputs details of the start, end and
 length of the input data set.}
 \item{\strong{$Fillgaps: }}{outputs the extent of missing data (years) in
 the original record and the gap filling method used (where necessary).}
 \item{\strong{$Bootstrapping.Iterations: }}{outputs the number of iterations
 used to generate the respective standard deviations for error margins.}
 \item{\strong{$Changepoints: }}{outputs the number and time at which
 changepoints in the variance of the uncorrelated residuals occur (if any).
 Where changepoints are identified, block bootstrapping procedures are used
 with residuals quarantined between changepoints.}
   }
}
\description{
Isolate trend component from mean sea level records.
}
\details{
This is the key entry point to the package. This function deconstructs
annual average time series data into a trend and associated real-time velocities
and accelerations, filling necessary internal structures to facilitate all
other functions in this package. The trend is isloated using Singular Spectrum
Analysis, in particular, aggregating components whose low frequency band [0 to
0.01] exceed a threshold contribution of 75\%. Associated velocities and
accelerations are determined through the fitting of a cubic smoothing spline
to the trend with 1 degree of freedom per every 8 years of record length.
Refer Watson (2016a,b) for more detail.
}
\examples{
# -------------------------------------------------------------------------
# Isolate trend from Baltimore record, filling gaps with spline interpolation and
# 500 iterations. Use raw 'Balt.csv' data file. Note: ordinarily user would call
# 'File.csv' direct from working directory using the following sample code:
# s <- msl.trend('Balt.csv', fillgaps = 3, iter = 500, 'BALTIMORE, USA') # DONT RUN
# -------------------------------------------------------------------------

data(s) # msl.trend object from above-mentioned example
str(s) # check structure of msl.trend object
msl.plot(s, type=2) # check screen output of gapfilling and trend estimate


}
\references{
Watson, P.J., 2016a. Identifying the best performing time series
analytics for sea-level research. In: \emph{Time Series Analysis and
Forecasting, Contributions to Statistics}, ISBN 978-3-319-28725-6, Springer
International Publishing (in press).

Watson, P.J., 2016b. How to improve estimates of real-time acceleration in
the mean sea level signal. In: Vila-Concejo, A., Bruce, E., Kennedy, D.M.,
and McCarroll, R.J. (eds.), Proceedings of the 14th International Coastal
Symposium (Sydney, Australia). \emph{Journal of Coastal Research},
Special Issue, No. 75. Coconut Creek (Florida), ISSN 0749-0208 (in press).
}
\seealso{
\code{\link{msl.forecast}}, \code{\link{msl.plot}},
\code{\link{msl.pdf}}, \code{\link{summary}}, \code{\link{Balt}}, \code{\link{s}}.
}

