% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/local_prune.R
\name{multiscale.localPrune}
\alias{multiscale.localPrune}
\title{Multiscale MOSUM algorithm with localised pruning}
\usage{
multiscale.localPrune(
  x,
  G = bandwidths.default(length(x)),
  max.unbalance = 4,
  threshold = c("critical.value", "custom")[1],
  alpha = 0.1,
  threshold.function = NULL,
  criterion = c("eta", "epsilon")[1],
  eta = 0.4,
  epsilon = 0.2,
  rule = c("pval", "jump")[1],
  penalty = c("log", "polynomial")[1],
  pen.exp = 1.01,
  do.confint = FALSE,
  level = 0.05,
  N_reps = 1000,
  ...
)
}
\arguments{
\item{x}{input data (a \code{numeric} vector or an object of classes \code{ts} and \code{timeSeries})}

\item{G}{a vector of bandwidths, given as either integers less than \code{length(x)/2}, 
or numbers between \code{0} and \code{0.5} describing the moving sum bandwidths relative to \code{length(x)}.
Asymmetric bandwidths obtained as the Cartesian product of the set \code{G} with itself are used for change point analysis}

\item{max.unbalance}{a numeric value for the maximal ratio between maximal and minimal bandwidths to be used for candidate generation,
\code{1 <= max.unbalance <= Inf}}

\item{threshold}{string indicating which threshold should be used to determine significance.
By default, it is chosen from the asymptotic distribution at the significance level \code{alpha}.
Alternatively, it is possible to parse a user-defined function with \code{threshold.function}}

\item{alpha}{a numeric value for the significance level with
\code{0 <= alpha <= 1}. Use iff \code{threshold = "critical.value"}}

\item{threshold.function}{function object of form \code{function(G_l, G_r, length(x), alpha)}, to compute a
threshold of significance for different bandwidths \code{(G_l, G_r)}; use iff \code{threshold = "custom"}}

\item{criterion}{how to determine whether an exceeding point is a change point; to be parsed to \link[mosum]{mosum}}

\item{eta, epsilon}{see \link[mosum]{mosum}}

\item{rule}{string for the choice of sorting criterion for change point candidates in merging step. 
Possible values are: 
\itemize{
\item{\code{"pval"}}{smallest p-value}
\item{\code{"jump"}}{largest (rescaled) jump size}
}}

\item{penalty}{string specifying the type of penalty term to be used in Schwarz criterion; possible values are:
\itemize{
\item{\code{"log"}}{use \code{penalty = log(length(x))^pen.exp}}
\item{\code{"polynomial"}}{use \code{penalty = length(x)^pen.exp}}
}}

\item{pen.exp}{exponent for the penalty term (see \code{penalty});}

\item{do.confint}{flag indicating whether confidence intervals for change points should be computed}

\item{level}{use iff \code{do.confint = TRUE}; a numeric value (\code{0 <= level <= 1}) with which
\code{100(1-level)\%} confidence interval is generated}

\item{N_reps}{use iff \code{do.confint = TRUE}; number of bootstrap replicates to be generated}

\item{...}{further arguments to be parsed to \link[mosum]{mosum} calls}
}
\value{
S3 object of class \code{multiscale.cpts}, which contains the following fields:
   \item{x}{input data}
   \item{cpts}{estimated change points}
   \item{cpts.info}{data frame containing information about estimated change points}
   \item{sc}{Schwarz criterion values of the estimated change point set}
   \item{pooled.cpts}{set of change point candidates that have been considered by the algorithm}
   \item{G}{input parameter}
   \item{threshold, alpha, threshold.function}{input parameters}
   \item{criterion, eta, epsilon}{input parameters}
   \item{rule, penalty, pen.exp}{input parameters}
   \item{do.confint}{input parameter}
   \item{ci}{object of class \code{cpts.ci} containing confidence intervals for change points iff \code{do.confint = TRUE}}
}
\description{
Multiscale MOSUM procedure with (possibly) assymetric bandwidths and localised pruning based on Schwarz criterion.
}
\details{
See Algorithm 2 in the first referenced paper for a comprehensive
description of the procedure and further details.
}
\examples{
x <- testData(model = "mix", seed = 123)$x
mlp <- multiscale.localPrune(x, G = c(8, 15, 30, 70), do.confint = TRUE)
print(mlp)
summary(mlp)
par(mfcol=c(2, 1), mar = c(2, 4, 2, 2))
plot(mlp, display = "data", shaded = "none")
plot(mlp, display = "significance", shaded = "CI", CI = "unif")
}
\references{
A. Meier, C. Kirch and H. Cho (2021)
mosum: A Package for Moving Sums in Change-point Analysis.
\emph{Journal of Statistical Software}, Volume 97, Number 8, pp. 1-42.
<doi:10.18637/jss.v097.i08>.

H. Cho and C. Kirch (2022) Two-stage data segmentation permitting multiscale change points, heavy tails and dependence. \emph{Annals of the Institute of Statistical Mathematics}, Volume 74, Number 4, pp. 653-684.

H. Cho and C. Kirch (2022) Bootstrap confidence intervals for multiple change points based on moving sum procedures. \emph{Computational Statistics & Data Analysis}, Volume 175, pp. 107552.
}
