% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/function_fordistribution.R
\name{d_outbreakdistn}
\alias{d_outbreakdistn}
\alias{p_outbreakdistn}
\alias{q_outbreakdistn}
\alias{r_outbreakdistn}
\title{The final outbreak size distribution}
\usage{
d_outbreakdistn(
  x = 10,
  seed.size = 1,
  epi.para = list(mean = 1, disp = 0.5, shift = 0.2),
  offspring.type = "D",
  is.log = FALSE
)

p_outbreakdistn(
  q = 30.5,
  seed.size = 1,
  epi.para = list(mean = 1, disp = 0.5, shift = 0.2),
  offspring.type = "D",
  lower.tail = TRUE,
  is.log = FALSE
)

q_outbreakdistn(
  p = 0.8,
  seed.size = 1,
  epi.para = list(mean = 1, disp = 0.5, shift = 0.2),
  offspring.type = "D",
  lower.tail = TRUE,
  upr.limit = 1000
)

r_outbreakdistn(
  n = 10,
  seed.size = 1,
  epi.para = list(mean = 1, disp = 0.5, shift = 0.2),
  offspring.type = "D",
  upr.limit = 1000
)
}
\arguments{
\item{x}{A scalar, or a vector of final outbreak size, which is positive integer. The value of \code{x} must be not less than \code{seed.size}.}

\item{seed.size}{A scalar, or a vector of positive integer.
For vector type of \code{seed.size}, it only applies to \code{d_outbreakdistn()}, and \code{p_outbreakdistn()}.
If \code{seed.size} and \code{x} or \code{q} are vectors, \code{seed.size} should be of the same length as \code{x} or \code{q}.}

\item{epi.para}{A list (\code{list}) of pre-defined epidemiological parameters for offspring distribution, in the format of \code{list(mean = ?, disp = ?, shift = ?)},
where the three parameters accept non-negative values.
Each parameter must be a scalar.
For Delaporte distribution, the value of \code{mean} should be larger than the value of \code{shift}.}

\item{offspring.type}{A character label (\code{character}) indicating the type of distribution used to describe the offspring distribution.
It only accepts one of the following values:
\itemize{
\item{\code{"D"}}{ indicates the \emph{Delaporte} distribution, }
\item{\code{"NB"}}{ indicates the \emph{negative binomial} distribution, }
\item{\code{"G"}}{ indicates the \emph{geometric} distribution, or }
\item{\code{"P"}}{ indicates the \emph{Poisson} distribution. }
}
By default, \code{offspring.type = 'D'}.}

\item{is.log}{A logical variable, under which probability would be taken natural logarithm, if \code{is.log = TRUE}.
By default, \code{is.log = FALSE}.}

\item{q}{A scalar, or a vector of positive number (not necessarily integer). The value of \code{q} must be not less than \code{seed.size}.}

\item{lower.tail}{A logical variable, under which the probability is cumulative distribution function (CDF, i.e., \emph{P(X <= x)}), if \code{lower.tail = TRUE}, and otherwise, 1 - CDF (i.e., \emph{P(X > x)}).
By default, \code{lower.tail = TRUE}.}

\item{p}{A scalar, or a vector of probability (i.e., ranging from 0 to 1).}

\item{upr.limit}{A positive integer.
If the result final outbreak size is larger than \code{upr.limit}, the returned value will be set as "\code{> upr.limit}".
The value of \code{upr.limit} must be larger than \code{seed.size}.
By default, \code{upr.limit = 1000}, and no need to change the default setting here unless for special reasons.}

\item{n}{A scalar of positive integer.}
}
\value{
For the values returned from the four functions,
\itemize{
\item{\code{d_outbreakdistn()}}{ is the \emph{probability mass function} (PMF), and it returns value of probability (i.e., ranging from 0 to 1); }
\item{\code{p_outbreakdistn()}}{ is the \emph{cumulative distribution function} (CDF), and it returns value of probability (i.e., ranging from 0 to 1); }
\item{\code{q_outbreakdistn()}}{ is the \emph{quantile function}, and it returns value of quantile (positive integer); and }
\item{\code{r_outbreakdistn()}}{ is the \emph{random variable generating function}, and it generates a set of \code{n} random variables (positive integers). }
}

Specially, due to the computational consumption, an upper limit was set for the functions \code{q_outbreakdistn()} and \code{r_outbreakdistn()},
i.e., \code{upr.limit}, and thus, both functions here return value in string form (i.e., \code{character}).
}
\description{
Density, cumulative distribution, quantile, and random variable generating functions for the \strong{final outbreak size} distribution with pre-defined epidemiological parameters.
}
\details{
Function \code{d_outbreakdistn()} returns the probability of having an outbreak with final size \code{x} generated by \code{seed.size} index cases, where (\code{seed.size}) is given.

Function \code{p_outbreakdistn()} returns the probability of having an outbreak with final size less than or equal to, or larger than \code{q} (depending on the value of \code{lower.tail}), generated by \code{seed.size} index cases, where (\code{seed.size}) is given.

Function \code{q_outbreakdistn()} returns a value such that there is a probability of \code{p} for having a final outbreak size less than or equal to, or larger than this value (depending on the value of \code{lower.tail}) generated by \code{seed.size} index cases, where (\code{seed.size}) is given.

Function \code{r_outbreakdistn()} returns a set of \code{n} random variables of final outbreak size, given (\code{seed.size}).
}
\note{
Each parameter in \code{epi.para = list(mean = ?, disp = ?, shift = ?)} should be a scalar, which means vector is not allowed here.

When \code{q} is large, e.g., \code{q} > 10000, the function \code{p_outbreakdistn()} could take few seconds, or even minutes to complete.

When \code{upr.limit} is large, e.g., \code{upr.limit} > 10000, the functions \code{q_outbreakdistn()} and \code{r_outbreakdistn()} could take few seconds, or even minutes to complete.
Thus, we do not recommend the users to change the default setting of \code{upr.limit} unless for special reasons.
}
\examples{

## an example to generate 1000 rv of final outbreak size
table(r_outbreakdistn(
  n = 1000,
  seed.size = 1,
  epi.para = list(mean = 1, disp = 0.5, shift = 0.2),
  offspring.type = "D",
  upr.limit = 10
))


\donttest{

## an attempt to reproduce the results in Guo, et al. (2022)
## paper doi link: https://doi.org/10.1016/j.jinf.2022.05.041 (see Fig 1B),
## where the probability of one seed case generating an outbreak with final size >= a given number,
## with parameter R of 0.78 and k of 0.10 under NB distribution.
plot(1:100, 1 - c(0,cumsum(d_outbreakdistn(
  x = 1:99,
  seed.size = 1,
  epi.para = list(mean = 0.78, disp = 0.10, shift = 0.2),
  offspring.type = "NB",
))), log = 'y', type = 'l', xlab = 'outbreak size', ylab = 'probability')
plot(1:100, c(1,p_outbreakdistn(
  q = 1:99,
  seed.size = 1,
  epi.para = list(mean = 0.78, disp = 0.10, shift = 0.2),
  offspring.type = "NB",
  lower.tail = FALSE
)), log = 'y', type = 'l', xlab = 'outbreak size', ylab = 'probability')

}

}
\references{
Farrington CP, Kanaan MN, Gay NJ. Branching process models for surveillance of infectious diseases controlled by mass vaccination. \emph{Biostatistics}. 2003;4(2):279-95.
\doi{10.1093/biostatistics/4.2.279}

Nishiura H, Yan P, Sleeman CK, Mode CJ. Estimating the transmission potential of supercritical processes based on the final size distribution of minor outbreaks. \emph{Journal of Theoretical Biology}. 2012;294:48-55.
\doi{10.1016/j.jtbi.2011.10.039}

Blumberg S, Funk S, Pulliam JR. Detecting differential transmissibilities that affect the size of self-limited outbreaks. \emph{PLoS Pathogens}. 2014;10(10):e1004452.
\doi{10.1371/journal.ppat.1004452}

Kucharski AJ, Althaus CL. The role of superspreading in Middle East respiratory syndrome coronavirus (MERS-CoV) transmission. \emph{Eurosurveillance}. 2015;20(25):21167.
\doi{10.2807/1560-7917.ES2015.20.25.21167}

Endo A, Abbott S, Kucharski AJ, Funk S. Estimating the overdispersion in COVID-19 transmission using outbreak sizes outside China. \emph{Wellcome Open Research}. 2020;5:67.
\doi{10.12688/wellcomeopenres.15842.3}

Zhao S, Chong MK, Ryu S, Guo Z, He M, Chen B, Musa SS, Wang J, Wu Y, He D, Wang MH. Characterizing superspreading potential of infectious disease: Decomposition of individual transmissibility. \emph{PLoS Computational Biology}. 2022;18(6):e1010281.
\doi{10.1371/journal.pcbi.1010281}
}
\seealso{
\code{\link[modelSSE:d_offspringdistn]{d_offspringdistn}}
}
