% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/missRanger.R
\name{missRanger}
\alias{missRanger}
\title{Fast Imputation of Missing Values by Chained Random Forests}
\usage{
missRanger(
  data,
  formula = . ~ .,
  pmm.k = 0L,
  num.trees = 500,
  mtry = NULL,
  min.node.size = NULL,
  min.bucket = NULL,
  max.depth = NULL,
  replace = TRUE,
  sample.fraction = if (replace) 1 else 0.632,
  case.weights = NULL,
  num.threads = NULL,
  save.memory = FALSE,
  maxiter = 10L,
  seed = NULL,
  verbose = 1,
  returnOOB = FALSE,
  data_only = !keep_forests,
  keep_forests = FALSE,
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} with missing values to impute.}

\item{formula}{A two-sided formula specifying variables to be imputed
(left hand side) and variables used to impute (right hand side).
Defaults to \code{. ~ .}, i.e., use all variables to impute all variables.
For instance, if all variables (with missings) should be imputed by all variables
except variable "ID", use \code{. ~ . - ID}. Note that a "." is evaluated
separately for each side of the formula. Further note that variables with missings
must appear in the left hand side if they should be used on the right hand side.}

\item{pmm.k}{Number of candidate non-missing values to sample from in the
predictive mean matching steps. 0 to avoid this step.}

\item{num.trees}{Number of trees passed to \code{\link[ranger:ranger]{ranger::ranger()}}.}

\item{mtry}{Number of covariates considered per split. The default \code{NULL} equals
the rounded down root of the number of features. Can be a function, e.g.,
\code{function(p) trunc(p/3)}. Passed to \code{\link[ranger:ranger]{ranger::ranger()}}. Note that during the
first iteration, the number of features is growing. Thus, a fixed value can lead to
an error. Using a function like \code{function(p) min(p, 2)} will fix such problem.}

\item{min.node.size}{Minimal node size passed to \code{\link[ranger:ranger]{ranger::ranger()}}.
By default 1 for classification and 5 for regression.}

\item{min.bucket}{Minimal terminal node size passed to \code{\link[ranger:ranger]{ranger::ranger()}}.
The default \code{NULL} means 1.}

\item{max.depth}{Maximal tree depth passed to \code{\link[ranger:ranger]{ranger::ranger()}}.
\code{NULL} means unlimited depth. 1 means single split trees.}

\item{replace}{Sample with replacement passed to \code{\link[ranger:ranger]{ranger::ranger()}}.}

\item{sample.fraction}{Fraction of rows per tree passed to \code{\link[ranger:ranger]{ranger::ranger()}}.
The default: use all rows when \code{replace = TRUE} and 0.632 otherwise.}

\item{case.weights}{Optional case weights passed to \code{\link[ranger:ranger]{ranger::ranger()}}.}

\item{num.threads}{Number of threads passed to \code{\link[ranger:ranger]{ranger::ranger()}}.
The default \code{NULL} uses all threads.}

\item{save.memory}{Slow but memory saving mode of \code{\link[ranger:ranger]{ranger::ranger()}}.}

\item{maxiter}{Maximum number of iterations.}

\item{seed}{Integer seed.}

\item{verbose}{A value in 0, 1, 2 controlling the verbosity.}

\item{returnOOB}{Should the final average OOB prediction errors be added
as data attribute "oob"? Only relevant when \code{data_only = TRUE}.}

\item{data_only}{If \code{TRUE} (default), only the imputed data is returned.
Otherwise, a "missRanger" object with additional information is returned.}

\item{keep_forests}{Should the random forests of the last relevant iteration
be returned? The default is \code{FALSE}. Setting this option will use a lot of memory.
Only relevant when \code{data_only = TRUE}.}

\item{...}{Additional arguments passed to \code{\link[ranger:ranger]{ranger::ranger()}}. Not all make sense.}
}
\value{
If \code{data_only = TRUE} an imputed \code{data.frame}. Otherwise, a "missRanger" object
with the following elements:
\itemize{
\item \code{data}: The imputed data.
\item \code{data_raw}: The original data provided.
\item \code{forests}: When \code{keep_forests = TRUE}, a list of "ranger" models used to
generate the imputed data. \code{NULL} otherwise.
\item \code{to_impute}: Variables to be imputed (in this order).
\item \code{impute_by}: Variables used for imputation.
\item \code{best_iter}: Best iteration.
\item \code{pred_errors}: Per-iteration OOB prediction errors (1 - R^2 for regression,
classification error otherwise).
\item \code{mean_pred_errors}: Per-iteration averages of OOB prediction errors.
\item \code{pmm.k}: Same as input \code{pmm.k}.
}
}
\description{
Uses the "ranger" package (Wright & Ziegler) to do fast missing value imputation by
chained random forests, see Stekhoven & Buehlmann and Van Buuren & Groothuis-Oudshoorn.
Between the iterative model fitting, it offers the option of predictive mean matching.
This firstly avoids imputation with values not present in the original data
(like a value 0.3334 in a 0-1 coded variable).
Secondly, predictive mean matching tries to raise the variance in the resulting
conditional distributions to a realistic level. This allows to do multiple imputation
when repeating the call to \code{\link[=missRanger]{missRanger()}}.
}
\details{
The iterative chaining stops as soon as \code{maxiter} is reached or if the average
out-of-bag (OOB) prediction errors stop reducing.
In the latter case, except for the first iteration, the second last (= best)
imputed data is returned.

OOB prediction errors are quantified as 1 - R^2 for numeric variables, and as
classification error otherwise. If a variable has been imputed only univariately,
the value is 1.
}
\examples{
iris2 <- generateNA(iris, seed = 1)

imp1 <- missRanger(iris2, pmm.k = 5, num.trees = 50, seed = 1)
head(imp1)

# Extended output
imp2 <- missRanger(iris2, pmm.k = 5, num.trees = 50, data_only = FALSE, seed = 1)
summary(imp2)

all.equal(imp1, imp2$data)

# Formula interface: Univariate imputation of Species and Sepal.Width
imp3 <- missRanger(iris2, Species + Sepal.Width ~ 1)
}
\references{
\enumerate{
\item Wright, M. N. & Ziegler, A. (2016). ranger: A Fast Implementation of
Random Forests for High Dimensional Data in C++ and R. Journal of Statistical
Software, in press. <arxiv.org/abs/1508.04409>.
\item Stekhoven, D.J. and Buehlmann, P. (2012). 'MissForest - nonparametric missing
value imputation for mixed-type data', Bioinformatics, 28(1) 2012, 112-118.
https://doi.org/10.1093/bioinformatics/btr597.
\item Van Buuren, S., Groothuis-Oudshoorn, K. (2011). mice: Multivariate Imputation
by Chained Equations in R. Journal of Statistical Software, 45(3), 1-67.
http://www.jstatsoft.org/v45/i03/
}
}
