% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findNextCATItem.R
\name{findNextItem}
\alias{findNextItem}
\title{Find next CAT item}
\usage{
findNextItem(
  x,
  person = NULL,
  test = NULL,
  design = NULL,
  criteria = NULL,
  objective = NULL,
  subset = NULL,
  all_index = FALSE,
  ...
)
}
\arguments{
\item{x}{an object of class 'mirtCAT_design' returned from the \code{\link{mirtCAT}} function
when passing \code{design_elements = TRUE}}

\item{person}{(required when \code{x} is missing) internal person object. To be
used when \code{customNextItem} function has been defined}

\item{test}{(required when \code{x} is missing) internal test object. To be
used when \code{customNextItem} function has been defined}

\item{design}{(required when \code{x} is missing) internal design object. To be
used when \code{customNextItem} function has been defined}

\item{criteria}{item selection criteria (see \code{\link{mirtCAT}}'s \code{criteria} input).
If not specified the value from \code{extract.mirtCAT(design, 'criteria')} will be used}

\item{objective}{a vector of values used as the optimization criteria to be passed to
\code{lp(objective.in)}. This is typically the vector of criteria values returned from
\code{\link{computeCriteria}}, however supplying other
criteria are possible (e.g., to minimize the number of items administered simply pass a vector
of -1's)}

\item{subset}{an integer vector indicating which items should be included in the optimal search;
the default \code{NULL} includes all possible items. To allow only the first 10 items to be
selected from this can be modified to \code{subset = 1:10}. This is useful when administering
a multi-unidimensional CAT session where unidimensional blocks should be clustered together
for smoother presentation. Useful when using the \code{customNextItem} function in
\code{\link{mirtCAT}}}

\item{all_index}{logical; return all items instead of just the most optimal?
When \code{TRUE} a vector of items is returned instead of the most optimal,
where the items are sorted according to how
well they fit the criteria (e.g., the first element is the most optimal, followed by the second
most optimal, and so on). Note that this does not work for some selection criteria (e.g.,
'seq' or 'random')}

\item{...}{additional arguments to be passed to \code{\link[lpSolve]{lp}}}
}
\value{
typically returns an integer value indicating the index of the next item to be selected or a
  value of \code{NA} to indicate that the test should be terminated. However, see the arguments for
  further returned object descriptions
}
\description{
A function that returns the next item in the computerized adaptive, optimal assembly, or shadow test.
For direction manipulation of the internal objects this function should be used in conjunction
with the \code{\link{updateDesign}} and \code{customNextItem}.
Finally, the raw input forms can be used when a \code{customNextItem} function has been
defined in \code{\link{mirtCAT}}.
}
\details{
When a numeric \code{objective} is supplied the next item in the computerized adaptive test is found via
an integer solver through searching for a maximum. The raw input forms can be used
when a \code{customNextItem} function has been defined in \code{\link{mirtCAT}}, and requires
the definition of a \code{constr_fun} (see the associated element in \code{\link{mirtCAT}} for details,
as well as the examples below). Can be used to for 'Optimal Test Assembly',
as well as 'Shadow Testing' designs (van der Linden, 2005),
by using the \code{\link[lpSolve]{lp}} function. When \code{objective} is not supplied the result follows the
typical maximum criteria of more standard adaptive tests.
}
\examples{
\dontrun{

# test defined in mirtCAT help file, first example 
# equivalent to criteria = 'MI'
customNextItem <- function(design, person, test){
   item <- findNextItem(person=person, design=design, test=test,
                        criteria = 'MI')
   item
 }
 
set.seed(1)
nitems <- 100
itemnames <- paste0('Item.', 1:nitems)
a <- matrix(rlnorm(nitems, .2, .3))
d <- matrix(rnorm(nitems))
dat <- simdata(a, d, 500, itemtype = 'dich')
colnames(dat) <- itemnames
mod <- mirt(dat, 1, verbose = FALSE)

# simple math items
questions <- answers <- character(nitems)
choices <- matrix(NA, nitems, 5)
spacing <- floor(d - min(d)) + 1 #easier items have more variation in the options

for(i in 1:nitems){
 n1 <- sample(1:50, 1)
 n2 <- sample(51:100, 1)
 ans <- n1 + n2
 questions[i] <- paste0(n1, ' + ', n2, ' = ?')
 answers[i] <- as.character(ans)
 ch <- ans + sample(c(-5:-1, 1:5) * spacing[i,], 5)
 ch[sample(1:5, 1)] <- ans
 choices[i, ] <- as.character(ch)
}

df <- data.frame(Question=questions, Option=choices, 
              Type = 'radio', stringsAsFactors = FALSE)
   
response <- generate_pattern(mod, 1)
result <- mirtCAT(mo=mod, local_pattern = response, 
                  design = list(customNextItem=customNextItem))
                
-----------------------------------------------------------
# direct manipulation of internal objects
CATdesign <- mirtCAT(df=df, mo=mod, criteria = 'MI', design_elements = TRUE)

# returns number 1 in this case, since that's the starting item
findNextItem(CATdesign)

# determine next item if item 1 and item 10 were answered correctly
CATdesign <- updateDesign(CATdesign, new_item = 1, new_response = 1)
extract.mirtCAT(CATdesign$person, 'thetas') # updated thetas
CATdesign <- updateDesign(CATdesign, new_item = 10, new_response = 1)
extract.mirtCAT(CATdesign$person, 'thetas') # updated thetas again
findNextItem(CATdesign)
findNextItem(CATdesign, all_index = TRUE) # all items rank in terms of most optimal

#-------------------------------------------------------------
## Integer programming example (e.g., shadow testing)

# find maximum information subject to constraints
#  sum(xi) <= 5               ### 5 or fewer items
#  x1 + x2 <= 1               ### items 1 and 2 can't be together
#  x4 == 0                    ### item 4 not included
#  x5 + x6 == 1               ### item 5 or 6 must be included, but not both

# constraint function
constr_fun <- function(design, person, test){

  # left hand side constrains
  #    - 1 row per constraint, and ncol must equal number of items
  mo <- extract.mirtCAT(test, 'mo')
  nitems <- extract.mirt(mo, 'nitems')
  lhs <- matrix(0, 4, nitems)
  lhs[1,] <- 1
  lhs[2,c(1,2)] <- 1
  lhs[3, 4] <- 1
  lhs[4, c(5,6)] <- 1

  # relationship direction
  dirs <- c("<=", "<=", '==', '==')

  #right hand side
  rhs <- c(5, 1, 0, 1)

  #all together
  constraints <- data.frame(lhs, dirs, rhs)
  constraints
}

CATdesign <- mirtCAT(df=df, mo=mod, design_elements = TRUE,
                     design = list(constr_fun=constr_fun))

# MI criteria value associated with each respective item
objective <- computeCriteria(CATdesign, criteria = 'MI')

# most optimal item, given constraints
findNextItem(CATdesign, objective=objective)

# all the items which solve the problem
findNextItem(CATdesign, objective=objective, all_index = TRUE)

## within a customNextItem() definition the above code would look like
# customNextItem <- function(design, person, test){
#   objective <- computeCriteria(person=person, design=design, test=test,
#                                criteria = 'MI')
#   item <- findNextItem(person=person, design=design, test=test,
#                        objective=objective)
#   item
# }

}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory
Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.
\doi{10.18637/jss.v048.i06}

Chalmers, R. P. (2016). Generating Adaptive and Non-Adaptive Test Interfaces for 
Multidimensional Item Response Theory Applications. \emph{Journal of Statistical Software, 71}(5), 
1-39. \doi{10.18637/jss.v071.i05}

van der Linden, W. J. (2005). Linear models for optimal test design. Springer.
}
\seealso{
\code{\link{mirtCAT}}, \code{\link{updateDesign}}, \code{\link{extract.mirtCAT}}
}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
