% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/miic.plot.R
\name{export}
\alias{export}
\title{Export miic result for plotting (with igraph)}
\usage{
export(
  miic_obj,
  method = "igraph",
  pcor_palette = NULL,
  display = "compact",
  show_self_loops = TRUE
)
}
\arguments{
\item{miic_obj}{[a miic object, required]

The object returned by the \code{\link{miic}} execution.}

\item{method}{[a string, optional, default value "igraph"]

The plotting method, currently only "igraph" is supported.}

\item{pcor_palette}{[a color palette, optional, default value
grDevices::colorRampPalette(c("blue", "darkgrey", "red")]

Used to represent the partial correlations (the color of the edges).
The palette must be able to handle 201 shades to cover the correlation range
from -100 to +100.}

\item{display}{[a string, optional, default value "compact"]

Used only when exporting object returned by miic in temporal mode.
It allows different representations of the temporal graph.
Possible values are \emph{"raw"}, \emph{"lagged"}, \emph{"compact"},
\emph{"combine"}, \emph{"unique"}, \emph{"drop"}:
\itemize{
\item When \emph{display} = \emph{"raw"}, the export function will
  use the tmiic graph object as it, leading to the return of a lagged
  graph.
\item When \emph{display} = \emph{"lagged"}, the export function will
  repeat the edges over history assuming stationarity and return a lagged
  graph.
\item When \emph{display} = \emph{"compact"}, the default, nodes
  and edges are converted into a flattened version to produce a compact
  view of the temporal network whilst still presenting all the information
  in the export.\cr
  e.g. X_lag1->Y_lag0, X_lag2<-Y_lag0 become respectively X->Y lag=1,
  X<-Y lag=2.
\item When \emph{display} = \emph{"combine"}, prior to the export,
  a pre-processing will be applied to kept only one edge
  per pair of nodes. The info_shifted will be the highest one
  of the summarized edges whilst the lag and orientation of the
  summarized edge will be an aggregation.\cr
  e.g. X_lag2->Y_lag0, X_lag0<-Y_lag1 will become X<->Y lag=1-2 with
  the info_shifted of X_lag2->Y_lag0 if info_shifted of
  X_lag2->Y_lag0 > X_lag0<-Y_lag1.
\item When \emph{display} = \emph{"unique"}, prior to the export,
  a pre-processing will be applied to kept only the edges having the
  highest info_shifted for a pair of nodes.
  If several edges between the sames nodes have the same
  info_shifted, then the edge kept is the one with the minimum lag.\cr
  e.g. X_lag1->Y_lag0, X_lag0<-Y_lag2 with info_shifted of
  X_lag1->Y_lag0 > X_lag0<-Y_lag2 become X->Y lag=1.
\item When \emph{display} = \emph{"drop"}, the same pre-processing
  as \emph{"unique"} will be applied, then the lag information will be
  dropped before the export.
}}

\item{show_self_loops}{[a boolean, optional, TRUE by default]

Used only when exporting object returned by miic in temporal mode.
When TRUE, the lagged edges starting and ending on the same node
are included in the igraph  object.
When FALSE, only edges having different nodes are present in the igraph
object.}
}
\value{
A graph object adapted to the method.
}
\description{
This function creates an object built from the result returned
by \code{\link{miic}} that is ready to be fed to the plotting method.
}
\details{
The behavior depends on the method used for the export.

For igraph, edge attributes are passed to the igraph graph
and can be accessed with e.g. \code{E(g)$partial_correlation}.
See \code{\link{miic}} for more details on edge parameters.
By default, edges are colored according to the partial correlation
between two nodes conditioned on the conditioning set
(negative is blue, null is gray and positive is red)
and their width is based on the conditional mutual information
minus the complexity cost.
}
\examples{
\donttest{
library(miic)
data(hematoData)

# execute MIIC (reconstruct graph)
miic_obj <- miic(
  input_data = hematoData, latent = "yes",
  n_shuffles = 10, conf_threshold = 0.001
)

# Using igraph
if(require(igraph)) {
g = export(miic_obj, "igraph")
plot(g) # Default visualisation, calls igraph::plot.igraph()

# Specifying layout (see ?igraph::layout_)
l <-layout_with_kk(g)
plot(g, layout=l)

# Override some graphical parameters
plot(g, edge.curved = .2)
plot(g, vertex.shape="none", edge.color="gray85", vertex.label.color="gray10")
}

# In temporal mode, execute MIIC
data(covidCases)
tmiic_obj <- miic(input_data = covidCases, mode = "TS", n_layers = 3, delta_t = 1, mov_avg = 14)

# Plot by default the compact display of the temporal network using igraph
if(require(igraph)) {
g = export (tmiic_obj)
plot(g)

# Plot the raw temporal network using igraph
g = export(tmiic_obj, display="raw")
plot(g)

# Plot the complete temporal network using igraph (completed by stationarity)
g = export(tmiic_obj, display="lagged")
plot(g)

# Specifying layout (see ?igraph::layout_)
l <- layout_on_grid(g, width = 5, height = 3, dim = 2)
plot(g, layout=l)

# For compact temporal display, please be aware that the rendering of
# igraph::plot.igraph() is not optimal when the graph contains
# multiple edges between the same nodes.
# So, the recommend way to plot a compact graph is to use tmiic plotting:
plot(tmiic_obj)
}

}
}
