% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mies_methods.R
\name{mies_survival_comma}
\alias{mies_survival_comma}
\title{Choose Survivors According to the "Mu , Lambda" ("Comma") Strategy}
\usage{
mies_survival_comma(inst, mu, survival_selector, n_elite, elite_selector, ...)
}
\arguments{
\item{inst}{(\code{\link[bbotk:OptimInstance]{OptimInstance}})\cr
Optimization instance to evaluate.}

\item{mu}{(\code{integer(1)})\cr
Population target size, non-negative integer.}

\item{survival_selector}{(\code{\link{Selector}})\cr
\code{\link{Selector}} operator that selects surviving individuals depending on configuration values
and objective results. When \code{survival_selector$operate()} is called, then objectives that
are being minimized are multiplied with -1 (through \code{\link{mies_get_fitnesses}}), since \code{\link{Selector}}s always try to maximize fitness.\cr
The \code{\link{Selector}} must be primed on \code{inst$search_space}; this \emph{includes} the "budget" component
when performing multi-fidelity optimization.\cr
The given \code{\link{Selector}} may \emph{not} return duplicates.}

\item{n_elite}{(\code{integer(1)})\cr
Number of individuals to carry over from previous generations. \code{n_elite} individuals will be selected
by \code{elite_selector}, while \code{mu - n_elite} will be selected by \code{survival_selector} from the most
recent generation. \code{n_elite} may be 0 (no elitism), in which case only individuals from the newest
generation survive. \code{n_elite} must be strictly smaller than \code{mu} to permit any optimization progress.}

\item{elite_selector}{(\code{\link{Selector}})\cr
\code{\link{Selector}} operator that selects "elites", i.e. surviving individuals from previous generations,
depending on configuration values
and objective results. When \code{elite_selector$operate()} is called, then objectives that
are being minimized are multiplied with -1 (through \code{\link[=mies_get_fitnesses]{mies_get_fitnesses()}}), since \code{\link{Selector}}s always try to maximize fitness.\cr
The \code{\link{Selector}} must be primed on \code{inst$search_space}; this \emph{includes} the "budget" component
when performing multi-fidelity optimization.\cr
The given \code{\link{Selector}} may \emph{not} return duplicates.}

\item{...}{(any)\cr
Ignored, for compatibility with other \verb{mies_survival_*} functions.}
}
\value{
invisible \code{\link[data.table:data.table]{data.table}}: The value of \code{inst$archive$data}, changed
in-place with \code{eol} set to the current generation for non-survivors.
}
\description{
Choose survivors during a MIES iteration using the "Comma" survival strategy, i.e.
selecting survivors from the latest generation only, using a \code{\link{Selector}} operator, and choosing
"elites" from survivors from previous generations using a different \code{\link{Selector}} operator.

When \code{n_elite} is greater than the number of alive individuals from previous generations,
then all these individuals from previous generations survive. In this case, it is
possible that more than \code{mu - n_elite} individuals from the current generation survive.
Similarly, when \code{mu} is greater
than the number of alive individuals from the last generation, then all these individuals survive.
}
\examples{
set.seed(1)
library("bbotk")
lgr::threshold("warn")

# Define the objective to optimize
objective <- ObjectiveRFun$new(
  fun = function(xs) {
    z <- exp(-xs$x^2 - xs$y^2) + 2 * exp(-(2 - xs$x)^2 - (2 - xs$y)^2)
    list(Obj = z)
  },
  domain = ps(x = p_dbl(-2, 4), y = p_dbl(-2, 4)),
  codomain = ps(Obj = p_dbl(tags = "maximize"))
)

# Get a new OptimInstance
oi <- OptimInstanceSingleCrit$new(objective,
  terminator = trm("evals", n_evals = 100)
)

mies_init_population(inst = oi, mu = 3)
# Usually the offspring is generated using mies_generate_offspring()
# Here shorter for demonstration purposes.
offspring = generate_design_random(oi$search_space, 3)$data
mies_evaluate_offspring(oi, offspring = offspring)

# State before: different generations of individuals. Alive individuals have
# 'eol' set to 'NA'.
oi$archive

s = sel("best")
s$prime(oi$search_space)
mies_survival_comma(oi, mu = 3, survival_selector = s,
  n_elite = 2, elite_selector = s)

# sel("best") lets only the best individuals survive.
# mies_survival_comma selects from new individuals (generation 2 in this case)
# and old individuals (all others) separately: n_elite = 2 from old,
# mu - n_elite = 1 from new.
# The surviving individuals have 'eol' set to 'NA'
oi$archive
}
\seealso{
Other mies building blocks: 
\code{\link{mies_evaluate_offspring}()},
\code{\link{mies_generate_offspring}()},
\code{\link{mies_get_fitnesses}()},
\code{\link{mies_init_population}()},
\code{\link{mies_select_from_archive}()},
\code{\link{mies_step_fidelity}()},
\code{\link{mies_survival_plus}()}
}
\concept{mies building blocks}
