#' @title Asignar nombres de divisones geográficas
#'
#' @description Asigna nombres de divisones geográficas en base a una variable de referencia.
#'
#' @import stringr
#' @import dplyr
#'
#' @param df objeto \code{data.frame}.
#' @param common_var Nombre de la variable con el código común a utilizar.
#' @param level Nivel de la variable de referencia: r para región, p para provincia, c para comuna y zc para zona censal.
#'
#' @return df con nombres de divisiones geográficas
#' @export namify
#'
#' @examples
#'  \donttest{
#'    data(c17_example)
#'    ismt <- full_ismt(c17_example, 10, 1) |> namify('ine17', 'c')
#'  }


namify <- function(df, common_var, level) {

  clone <- df
  id_region <- NULL
  id_provin <- NULL
  zona <- NULL

  # ---- Regiones ----

  rnames0 <- c('01_TARAPACA', '01_TARAPACA', '01_TARAPACA', '01_TARAPACA', '01_TARAPACA', '01_TARAPACA', '01_TARAPACA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '02_ANTOFAGASTA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '03_ATACAMA', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '04_COQUIMBO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '05_VALPARAISO', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '06_OHIGGINS', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '07_MAULE', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO', '08_BIOBIO')
  rnames1 <- c('09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '09_ARAUCANIA', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '10_LOS_LAGOS', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '11_AYSEN', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '12_MAGALLANES', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '13_METROPOLITANA', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '14_LOS_RIOS', '15_ARICA_PARINACOTA', '15_ARICA_PARINACOTA', '15_ARICA_PARINACOTA', '15_ARICA_PARINACOTA', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE', '16_NUBLE')
  region <- c(rnames0, rnames1)

  # ---- Provincias ----

  pnames0 <- c('IQUIQUE', 'EL_TAMARUGAL', 'EL_TAMARUGAL', 'EL_TAMARUGAL', 'IQUIQUE', 'EL_TAMARUGAL', 'EL_TAMARUGAL', 'ANTOFAGASTA', 'CALAMA', 'TOCOPILLA', 'ANTOFAGASTA', 'CALAMA', 'CALAMA', 'ANTOFAGASTA', 'ANTOFAGASTA', 'TOCOPILLA', 'VALLENAR', 'COPIAPO', 'CHANARAL', 'COPIAPO', 'CHANARAL', 'VALLENAR', 'VALLENAR', 'COPIAPO', 'VALLENAR', 'COQUIMBO', 'CHOAPA', 'OVALLE', 'COQUIMBO', 'CHOAPA', 'COQUIMBO', 'COQUIMBO', 'CHOAPA', 'OVALLE', 'OVALLE', 'COQUIMBO', 'OVALLE', 'OVALLE', 'CHOAPA', 'COQUIMBO', 'SAN_ANTONIO', 'PETORCA', 'LOS_ANDES', 'SAN_ANTONIO', 'VALPARAISO', 'SAN_FELIPE_DE_ACONCAGUA', 'VALPARAISO', 'SAN_ANTONIO', 'SAN_ANTONIO', 'QUILLOTA', 'ISLA_DE_PASCUA', 'VALPARAISO', 'QUILLOTA', 'QUILLOTA', 'PETORCA', 'MARGA_MARGA', 'SAN_FELIPE_DE_ACONCAGUA', 'LOS_ANDES', 'QUILLOTA', 'MARGA_MARGA', 'SAN_FELIPE_DE_ACONCAGUA', 'PETORCA', 'PETORCA', 'VALPARAISO', 'SAN_FELIPE_DE_ACONCAGUA', 'QUILLOTA', 'MARGA_MARGA', 'VALPARAISO', 'LOS_ANDES', 'SAN_ANTONIO', 'LOS_ANDES', 'SAN_FELIPE_DE_ACONCAGUA', 'SAN_FELIPE_DE_ACONCAGUA', 'SAN_ANTONIO', 'VALPARAISO', 'MARGA_MARGA', 'VALPARAISO', 'PETORCA', 'COLCHAGUA', 'COLCHAGUA', 'CACHAPOAL', 'CACHAPOAL', 'CACHAPOAL', 'CACHAPOAL', 'CACHAPOAL', 'CARDENAL_CARO', 'CACHAPOAL', 'CARDENAL_CARO', 'COLCHAGUA', 'CACHAPOAL', 'CACHAPOAL', 'CARDENAL_CARO', 'CACHAPOAL', 'COLCHAGUA', 'CARDENAL_CARO', 'CACHAPOAL', 'COLCHAGUA', 'CARDENAL_CARO', 'COLCHAGUA', 'CACHAPOAL', 'CACHAPOAL', 'CARDENAL_CARO', 'COLCHAGUA', 'COLCHAGUA', 'CACHAPOAL', 'CACHAPOAL', 'CACHAPOAL', 'CACHAPOAL', 'COLCHAGUA', 'CACHAPOAL', 'COLCHAGUA', 'CAUQUENES', 'CAUQUENES', 'LINARES', 'TALCA', 'TALCA', 'CURICO', 'TALCA', 'CURICO', 'CURICO', 'LINARES', 'LINARES', 'TALCA', 'CURICO', 'LINARES', 'TALCA', 'CAUQUENES', 'TALCA', 'CURICO', 'LINARES', 'TALCA', 'CURICO', 'CURICO', 'TALCA', 'LINARES', 'TALCA', 'TALCA', 'CURICO', 'CURICO', 'LINARES', 'LINARES', 'BIOBIO', 'BIOBIO', 'ARAUCO', 'BIOBIO', 'ARAUCO', 'CONCEPCION', 'CONCEPCION', 'ARAUCO', 'CONCEPCION', 'ARAUCO', 'CONCEPCION', 'CONCEPCION', 'CONCEPCION', 'BIOBIO', 'ARAUCO', 'ARAUCO')
  pnames1 <- c('BIOBIO', 'CONCEPCION', 'BIOBIO', 'BIOBIO', 'BIOBIO', 'CONCEPCION', 'BIOBIO', 'BIOBIO', 'CONCEPCION', 'BIOBIO', 'BIOBIO', 'CONCEPCION', 'CONCEPCION', 'ARAUCO', 'CONCEPCION', 'BIOBIO', 'BIOBIO', 'MALLECO', 'CAUTIN', 'CAUTIN', 'MALLECO', 'CAUTIN', 'MALLECO', 'CAUTIN', 'MALLECO', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'MALLECO', 'MALLECO', 'MALLECO', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'MALLECO', 'MALLECO', 'CAUTIN', 'CAUTIN', 'CAUTIN', 'MALLECO', 'MALLECO', 'CAUTIN', 'CAUTIN', 'CHILOE', 'LLANQUIHUE', 'CHILOE', 'PALENA', 'CHILOE', 'LLANQUIHUE', 'CHILOE', 'CHILOE', 'LLANQUIHUE', 'LLANQUIHUE', 'PALENA', 'PALENA', 'LLANQUIHUE', 'LLANQUIHUE', 'LLANQUIHUE', 'OSORNO', 'PALENA', 'LLANQUIHUE', 'OSORNO', 'LLANQUIHUE', 'CHILOE', 'OSORNO', 'OSORNO', 'CHILOE', 'CHILOE', 'CHILOE', 'CHILOE', 'OSORNO', 'OSORNO', 'OSORNO', 'AYSEN', 'GENERAL_CARRERA', 'AYSEN', 'CAPITAN_PRAT', 'COYHAIQUE', 'AYSEN', 'COYHAIQUE', 'CAPITAN_PRAT', 'GENERAL_CARRERA', 'CAPITAN_PRAT', 'ANTARTICA_CHILENA', 'MAGALLANES', 'ULTIMA_ESPERANZA', 'TIERRA_DEL_FUEGO', 'TIERRA_DEL_FUEGO', 'MAGALLANES', 'MAGALLANES', 'MAGALLANES', 'TIERRA_DEL_FUEGO', 'ULTIMA_ESPERANZA', 'MELIPILLA', 'MAIPO', 'MAIPO', 'SANTIAGO', 'SANTIAGO', 'CHACABUCO', 'SANTIAGO', 'MELIPILLA', 'SANTIAGO', 'TALAGANTE', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'TALAGANTE', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'CHACABUCO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'MELIPILLA', 'MELIPILLA', 'SANTIAGO', 'TALAGANTE', 'MAIPO', 'SANTIAGO', 'TALAGANTE', 'SANTIAGO', 'CORDILLERA', 'SANTIAGO', 'SANTIAGO', 'CORDILLERA', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'SANTIAGO', 'MAIPO', 'SANTIAGO', 'CORDILLERA', 'SANTIAGO', 'MELIPILLA', 'SANTIAGO', 'SANTIAGO', 'TALAGANTE', 'CHACABUCO', 'SANTIAGO', 'VALDIVIA', 'RANCO', 'RANCO', 'RANCO', 'VALDIVIA', 'VALDIVIA', 'VALDIVIA', 'VALDIVIA', 'VALDIVIA', 'VALDIVIA', 'RANCO', 'VALDIVIA', 'ARICA', 'ARICA', 'PARINACOTA', 'PARINACOTA', 'DIGUILLIN', 'DIGUILLIN', 'DIGUILLIN', 'ITATA', 'ITATA', 'PUNILLA', 'DIGUILLIN', 'ITATA', 'PUNILLA', 'DIGUILLIN', 'DIGUILLIN', 'ITATA', 'DIGUILLIN', 'ITATA', 'ITATA', 'PUNILLA', 'PUNILLA', 'DIGUILLIN', 'PUNILLA', 'ITATA', 'DIGUILLIN')
  provincia <- c(pnames0, pnames1)

  # ---- Comunas ----

  cnames0 <- c('ALTO_HOSPICIO', 'CAMINA', 'COLCHANE', 'HUARA', 'IQUIQUE', 'PICA', 'POZO_ALMONTE', 'ANTOFAGASTA', 'CALAMA', 'MARIA_ELENA', 'MEJILLONES', 'OLLAGUE', 'SAN_PEDRO_DE_ATACAMA', 'SIERRA_GORDA', 'TALTAL', 'TOCOPILLA', 'ALTO_DEL_CARMEN', 'CALDERA', 'CHANARAL', 'COPIAPO', 'DIEGO_DE_ALMAGRO', 'FREIRINA', 'HUASCO', 'TIERRA_AMARILLA', 'VALLENAR', 'ANDACOLLO', 'CANELA', 'COMBARBALA', 'COQUIMBO', 'ILLAPEL', 'LA_HIGUERA', 'LA_SERENA', 'LOS_VILOS', 'MONTE_PATRIA', 'OVALLE', 'PAIHUANO', 'PUNITAQUI', 'RIO_HURTADO', 'SALAMANCA', 'VICUNA', 'ALGARROBO', 'CABILDO', 'CALLE_LARGA', 'CARTAGENA', 'CASABLANCA', 'CATEMU', 'CONCON', 'EL_QUISCO', 'EL_TABO', 'HIJUELAS', 'ISLA_DE_PASCUA', 'JUAN_FERNANDEZ', 'LA_CALERA', 'LA_CRUZ', 'LA_LIGUA', 'LIMACHE', 'LLAY-LLAY', 'LOS_ANDES', 'NOGALES', 'OLMUE', 'PANQUEHUE', 'PAPUDO', 'PETORCA', 'PUCHUNCAVI', 'PUTAENDO', 'QUILLOTA', 'QUILPUE', 'QUINTERO', 'RINCONADA', 'SAN_ANTONIO', 'SAN_ESTEBAN', 'SAN_FELIPE', 'SANTA_MARIA', 'SANTO_DOMINGO', 'VALPARAISO', 'VILLA_ALEMANA', 'VINA_DEL_MAR', 'ZAPALLAR', 'CHEPICA', 'CHIMBARONGO', 'CODEGUA', 'COINCO', 'COLTAUCO', 'DONIHUE', 'GRANEROS', 'LA_ESTRELLA', 'LAS_CABRAS', 'LITUECHE', 'LOLOL', 'MACHALI', 'MALLOA', 'MARCHIGUE', 'MOSTAZAL', 'NANCAGUA', 'NAVIDAD', 'OLIVAR', 'PALMILLA', 'PAREDONES', 'PERALILLO', 'PEUMO', 'PICHIDEGUA', 'PICHILEMU', 'PLACILLA', 'PUMANQUE', 'QUINTA_DE_TILCOCO', 'RANCAGUA', 'RENGO', 'REQUINOA', 'SAN_FERNANDO', 'SAN_VICENTE_DE_TAGUA_TAGUA', 'SANTA_CRUZ', 'CAUQUENES', 'CHANCO', 'COLBUN', 'CONSTITUCION', 'CUREPTO', 'CURICO', 'EMPEDRADO', 'HUALANE', 'LICANTEN', 'LINARES', 'LONGAVI', 'MAULE', 'MOLINA', 'PARRAL', 'PELARCO', 'PELLUHUE', 'PENCAHUE', 'RAUCO', 'RETIRO', 'RIO_CLARO', 'ROMERAL', 'SAGRADA_FAMILIA', 'SAN_CLEMENTE', 'SAN_JAVIER', 'SAN_RAFAEL', 'TALCA', 'TENO', 'VICHUQUEN', 'VILLA_ALEGRE', 'YERBAS_BUENAS', 'ALTO_BIOBIO', 'ANTUCO', 'ARAUCO', 'CABRERO', 'CANETE', 'CHIGUAYANTE', 'CONCEPCION', 'CONTULMO', 'CORONEL', 'CURANILAHUE', 'FLORIDA', 'HUALPEN', 'HUALQUI', 'LAJA', 'LEBU', 'LOS_ALAMOS', 'LOS_ANGELES', 'LOTA', 'MULCHEN', 'NACIMIENTO', 'NEGRETE', 'PENCO', 'QUILACO', 'QUILLECO', 'SAN_PEDRO_DE_LA_PAZ', 'SAN_ROSENDO', 'SANTA_BARBARA', 'SANTA_JUANA', 'TALCAHUANO', 'TIRUA', 'TOME', 'TUCAPEL', 'YUMBEL', 'ANGOL', 'CARAHUE', 'CHOLCHOL', 'COLLIPULLI', 'CUNCO', 'CURACAUTIN', 'CURARREHUE', 'ERCILLA', 'FREIRE', 'GALVARINO', 'GORBEA', 'LAUTARO', 'LONCOCHE', 'LONQUIMAY')
  cnames1 <- c('LOS_SAUCES', 'LUMACO', 'MELIPEUCO', 'NUEVA_IMPERIAL', 'PADRE_LAS_CASAS', 'PERQUENCO', 'PITRUFQUEN', 'PUCON', 'PUERTO_SAAVEDRA', 'PUREN', 'RENAICO', 'TEMUCO', 'TEODORO_SCHMIDT', 'TOLTEN', 'TRAIGUEN', 'VICTORIA', 'VILCUN', 'VILLARRICA', 'ANCUD', 'CALBUCO', 'CASTRO', 'CHAITEN', 'CHONCHI', 'COCHAMO', 'CURACO_DE_VELEZ', 'DALCAHUE', 'FRESIA', 'FRUTILLAR', 'FUTALEUFU', 'HUALAIHUE', 'LLANQUIHUE', 'LOS_MUERMOS', 'MAULLIN', 'OSORNO', 'PALENA', 'PUERTO_MONTT', 'PUERTO_OCTAY', 'PUERTO_VARAS', 'PUQUELDON', 'PURRANQUE', 'PUYEHUE', 'QUEILEN', 'QUELLON', 'QUEMCHI', 'QUINCHAO', 'RIO_NEGRO', 'SAN_JUAN_DE_LA_COSTA', 'SAN_PABLO', 'AYSEN', 'CHILE_CHICO', 'CISNES', 'COCHRANE', 'COYHAIQUE', 'GUAITECAS', 'LAGO_VERDE', 'OHIGGINS', 'RIO_IBANEZ', 'TORTEL', 'CABO_DE_HORNOS', 'LAGUNA_BLANCA', 'NATALES', 'PORVENIR', 'PRIMAVERA', 'PUNTA_ARENAS', 'RIO_VERDE', 'SAN_GREGORIO', 'TIMAUKEL', 'TORRES_DEL_PAINE', 'ALHUE', 'BUIN', 'CALERA_DE_TANGO', 'CERRILLOS', 'CERRO_NAVIA', 'COLINA', 'CONCHALI', 'CURACAVI', 'EL_BOSQUE', 'EL_MONTE', 'ESTACION_CENTRAL', 'HUECHURABA', 'INDEPENDENCIA', 'ISLA_DE_MAIPO', 'LA_CISTERNA', 'LA_FLORIDA', 'LA_GRANJA', 'LA_PINTANA', 'LA_REINA', 'LAMPA', 'LAS_CONDES', 'LO_BARNECHEA', 'LO_ESPEJO', 'LO_PRADO', 'MACUL', 'MAIPU', 'MARIA_PINTO', 'MELIPILLA', 'NUNOA', 'PADRE_HURTADO', 'PAINE', 'PEDRO_AGUIRRE_CERDA', 'PENAFLOR', 'PENALOLEN', 'PIRQUE', 'PROVIDENCIA', 'PUDAHUEL', 'PUENTE_ALTO', 'QUILICURA', 'QUINTA_NORMAL', 'RECOLETA', 'RENCA', 'SAN_BERNARDO', 'SAN_JOAQUIN', 'SAN_JOSE_DE_MAIPO', 'SAN_MIGUEL', 'SAN_PEDRO', 'SAN_RAMON', 'SANTIAGO', 'TALAGANTE', 'TIL_TIL', 'VITACURA', 'CORRAL', 'FUTRONO', 'LA_UNION', 'LAGO_RANCO', 'LANCO', 'LOS_LAGOS', 'MAFIL', 'MARIQUINA', 'PAILLACO', 'PANGUIPULLI', 'RIO_BUENO', 'VALDIVIA', 'ARICA', 'CAMARONES', 'GENERAL_LAGOS', 'PUTRE', 'BULNES', 'CHILLAN', 'CHILLAN_VIEJO', 'COBQUECURA', 'COELEMU', 'COIHUECO', 'EL_CARMEN', 'NINHUE', 'NIQUEN', 'PEMUCO', 'PINTO', 'PORTEZUELO', 'QUILLON', 'QUIRIHUE', 'RANQUIL', 'SAN_CARLOS', 'SAN_FABIAN', 'SAN_IGNACIO', 'SAN_NICOLAS', 'TREHUACO', 'YUNGAY')
  comuna <- c(cnames0, cnames1)

  # ---- INE 2017 ----

  ine17 <- c('01107', '01402', '01403', '01404', '01101', '01405', '01401', '02101', '02201', '02302', '02102', '02202', '02203', '02103', '02104', '02301', '03302', '03102', '03201', '03101', '03202', '03303', '03304', '03103', '03301', '04103', '04202', '04302', '04102', '04201', '04104', '04101', '04203', '04303', '04301', '04105', '04304', '04305', '04204', '04106', '05602', '05402', '05302', '05603', '05102', '05702', '05103', '05604', '05605', '05503', '05201', '05104', '05502', '05504', '05401', '05802', '05703', '05301', '05506', '05803', '05704', '05403', '05404', '05105', '05705', '05501', '05801', '05107', '05303', '05601', '05304', '05701', '05706', '05606', '05101', '05804', '05109', '05405', '06302', '06303', '06102', '06103', '06104', '06105', '06106', '06202', '06107', '06203', '06304', '06108', '06109', '06204', '06110', '06305', '06205', '06111', '06306', '06206', '06307', '06112', '06113', '06201', '06308', '06309', '06114', '06101', '06115', '06116', '06301', '06117', '06310', '07201', '07202', '07402', '07102', '07103', '07301', '07104', '07302', '07303', '07401', '07403', '07105', '07304', '07404', '07106', '07203', '07107', '07305', '07405', '07108', '07306', '07307', '07109', '07406', '07110', '07101', '07308', '07309', '07407', '07408', '08314', '08302', '08202', '08303', '08203', '08103', '08101', '08204', '08102', '08205', '08104', '08112', '08105', '08304', '08201', '08206', '08301', '08106', '08305', '08306', '08307', '08107', '08308', '08309', '08108', '08310', '08311', '08109', '08110', '08207', '08111', '08312', '08313', '09201', '09102', '09121', '09202', '09103', '09203', '09104', '09204', '09105', '09106', '09107', '98108', '09109', '09205', '09206', '09207', '09110', '09111', '09112', '09113', '09114', '09115', '09116', '09208', '09209', '09101', '09117', '09118', '09210', '09211', '09119', '09120', '10202', '10102', '10201', '10401', '10203', '10103', '10204', '10205', '10104', '10105', '10402', '10403', '10107', '10106', '10108', '10301', '10404', '10101', '10302', '10109', '10206', '10303', '10304', '10207', '10208', '10209', '10210', '10305', '10306', '10307', '11201', '11401', '11202', '11301', '11101', '11203', '11102', '11302', '11402', '11303', '12201', '12102', '12401', '12301', '12302', '12101', '12103', '12104', '12303', '12402', '13502', '13402', '13403', '13102', '13103', '13301', '13104', '13503', '13105', '13602', '13106', '13107', '13108', '13603', '13109', '13110', '13111', '13112', '13113', '13302', '13114', '13115', '13116', '13117', '13118', '13119', '13504', '13501', '13120', '13604', '13404', '13121', '13605', '13122', '13202', '13123', '13124', '13201', '13125', '13126', '13127', '13128', '13401', '13129', '13203', '13130', '13505', '13131', '13101', '13601', '13303', '13132', '14102', '14202', '14201', '14203', '14103', '14104', '14105', '14106', '14107', '14108', '14204', '14101', '15101', '15102', '15202', '15201', '16102', '16101', '16103', '16202', '16203', '16302', '16104', '16204', '16303', '16105', '16106', '16205', '16107', '16201', '16206', '16301', '16304', '16108', '16305', '16207', '16109')

  # ---- INE 2012 ----

  ine12 <- c('01107', '01402', '01403', '01404', '01101', '01405', '01401', '02101', '02201', '02302', '02102', '02202', '02203', '02103', '02104', '02301', '03302', '03102', '03201', '03101', '03202', '03303', '03304', '03103', '03301', '04103', '04202', '04302', '04102', '04201', '04104', '04101', '04203', '04303', '04301', '04105', '04304', '04305', '04204', '04106', '05602', '05402', '05302', '05603', '05102', '05702', '05103', '05604', '05605', '05503', '05201', '05104', '05502', '05504', '05401', '05802', '05703', '05301', '05506', '05803', '05704', '05403', '05404', '05105', '05705', '05501', '05801', '05107', '05303', '05601', '05304', '05701', '05706', '05606', '05101', '05804', '05109', '05405', '06302', '06303', '06102', '06103', '06104', '06105', '06106', '06202', '06107', '06203', '06304', '06108', '06109', '06204', '06110', '06305', '06205', '06111', '06306', '06206', '06307', '06112', '06113', '06201', '06308', '06309', '06114', '06101', '06115', '06116', '06301', '06117', '06310', '07201', '07202', '07402', '07102', '07103', '07301', '07104', '07302', '07303', '07401', '07403', '07105', '07304', '07404', '07106', '07203', '07107', '07305', '07405', '07108', '07306', '07307', '07109', '07406', '07110', '07101', '07308', '07309', '07407', '07408', '08314', '08302', '08202', '08303', '08203', '08103', '08101', '08204', '08102', '08205', '08104', '08112', '08105', '08304', '08201', '08206', '08301', '08106', '08305', '08306', '08307', '08107', '08308', '08309', '08108', '08310', '08311', '08109', '08110', '08207', '08111', '08312', '08313', '09201', '09102', '09121', '09202', '09103', '09203', '09104', '09204', '09105', '09106', '09107', '98108', '09109', '09205', '09206', '09207', '09110', '09111', '09112', '09113', '09114', '09115', '09116', '09208', '09209', '09101', '09117', '09118', '09210', '09211', '09119', '09120', '10202', '10102', '10201', '10401', '10203', '10103', '10204', '10205', '10104', '10105', '10402', '10403', '10107', '10106', '10108', '10301', '10404', '10101', '10302', '10109', '10206', '10303', '10304', '10207', '10208', '10209', '10210', '10305', '10306', '10307', '11201', '11401', '11202', '11301', '11101', '11203', '11102', '11302', '11402', '11303', '12201', '12102', '12401', '12301', '12302', '12101', '12103', '12104', '12303', '12402', '13502', '13402', '13403', '13102', '13103', '13301', '13104', '13503', '13105', '13602', '13106', '13107', '13108', '13603', '13109', '13110', '13111', '13112', '13113', '13302', '13114', '13115', '13116', '13117', '13118', '13119', '13504', '13501', '13120', '13604', '13404', '13121', '13605', '13122', '13202', '13123', '13124', '13201', '13125', '13126', '13127', '13128', '13401', '13129', '13203', '13130', '13505', '13131', '13101', '13601', '13303', '13132', '14102', '14202', '14201', '14203', '14103', '14104', '14105', '14106', '14107', '14108', '14204', '14101', '15101', '15102', '15202', '15201', '08402', '08401', '08406', '08403', '08404', '08405', '08407', '08408', '08409', '08410', '08411', '08412', '08413', '08414', '08415', '08416', '08417', '08418', '08419', '08420', '08421')

  # ---- INE 2002 ----

  ine02 <- c('01107', '01402', '01403', '01404', '01101', '01405', '01401', '02101', '02201', '02302', '02102', '02202', '02203', '02103', '02104', '02301', '03302', '03102', '03201', '03101', '03202', '03303', '03304', '03103', '03301', '04103', '04202', '04302', '04102', '04201', '04104', '04101', '04203', '04303', '04301', '04105', '04304', '04305', '04204', '04106', '05602', '05402', '05302', '05603', '05102', '05702', '05103', '05604', '05605', '05503', '05201', '05104', '05502', '05504', '05401', '05505', '05703', '05301', '05506', '05507', '05704', '05403', '05404', '05105', '05705', '05501', '05106', '05107', '05303', '05601', '05304', '05701', '05706', '05606', '05101', '05108', '05109', '05405', '06302', '06303', '06102', '06103', '06104', '06105', '06106', '06202', '06107', '06203', '06304', '06108', '06109', '06204', '06110', '06305', '06205', '06111', '06306', '06206', '06307', '06112', '06113', '06201', '06308', '06309', '06114', '06101', '06115', '06116', '06301', '06117', '06310', '07201', '07202', '07402', '07102', '07103', '07301', '07104', '07302', '07303', '07401', '07403', '07105', '07304', '07404', '07106', '07203', '07107', '07305', '07405', '07108', '07306', '07307', '07109', '07406', '07110', '07101', '07308', '07309', '07407', '07408', '08314', '08302', '08202', '08303', '08203', '08103', '08101', '08204', '08102', '08205', '08104', '08112', '08105', '08304', '08201', '08206', '08301', '08106', '08305', '08306', '08307', '08107', '08308', '08309', '08108', '08310', '08311', '08109', '08110', '08207', '08111', '08312', '08313', '09201', '09102', '09121', '09202', '09103', '09203', '09104', '09204', '09105', '09106', '09107', '98108', '09109', '09205', '09206', '09207', '09110', '09111', '09112', '09113', '09114', '09115', '09116', '09208', '09209', '09101', '09117', '09118', '09210', '09211', '09119', '09120', '10202', '10102', '10201', '10401', '10203', '10103', '10204', '10205', '10104', '10105', '10402', '10403', '10107', '10106', '10108', '10301', '10404', '10101', '10302', '10109', '10206', '10303', '10304', '10207', '10208', '10209', '10210', '10305', '10306', '10307', '11201', '11401', '11202', '11301', '11101', '11203', '11102', '11302', '11402', '11303', '12201', '12102', '12401', '12301', '12302', '12101', '12103', '12104', '12303', '12402', '13502', '13402', '13403', '13102', '13103', '13301', '13104', '13503', '13105', '13602', '13106', '13107', '13108', '13603', '13109', '13110', '13111', '13112', '13113', '13302', '13114', '13115', '13116', '13117', '13118', '13119', '13504', '13501', '13120', '13604', '13404', '13121', '13605', '13122', '13202', '13123', '13124', '13201', '13125', '13126', '13127', '13128', '13401', '13129', '13203', '13130', '13505', '13131', '13101', '13601', '13303', '13132', '14102', '14202', '14201', '14203', '14103', '14104', '14105', '14106', '14107', '14108', '14204', '14101', '15101', '15102', '15202', '15201', '08402', '08401', '08406', '08403', '08404', '08405', '08407', '08408', '08409', '08410', '08411', '08412', '08413', '08414', '08415', '08416', '08417', '08418', '08419', '08420', '08421')

  # ---- data.frame() ----

  ref <- data.frame(region, provincia, comuna, ine17, ine12, ine02) |>
    dplyr::mutate(
      id_region = stringr::str_sub(ine17, 1, 2),
      id_provin = stringr::str_sub(ine17, 1, 3),
      region = stringr::str_sub(region, 4, -1)
    ) |>
    dplyr::select(
      region, id_region, provincia, id_provin, comuna, ine17, ine12, ine02
    )

  # ---- Function ----

  if (level == 'r') {

    names(clone)[names(clone) == stringr::str_glue('{common_var}')] <- 'id_region'
    clone <- clone |>
      dplyr::mutate(
        id_region = stringr::str_pad(id_region, 2, 'left', '0')
      )
    ref <- ref |> dplyr::select(region, id_region)
    named <- merge(ref, clone, by = 'id_region')

  } else if (level == 'p') {

    names(clone)[names(clone) == stringr::str_glue('{common_var}')] <- 'id_provin'
    clone <- clone |>
      dplyr::mutate(
        id_provin = stringr::str_pad(id_provin, 3, 'left', '0')
      )
    ref <- ref |> dplyr::select(region, id_region, provincia, id_provin)
    named <- merge(ref, clone, by = 'id_provin')

  } else if (level == 'c') {

    names(clone)[names(clone) == stringr::str_glue('{common_var}')] <- 'ine17'
    clone <- clone |>
      dplyr::mutate(
        ine17 = stringr::str_pad(ine17, 5, 'left', '0')
      )
    ref <- ref |> dplyr::select(region, id_region, provincia, id_provin, comuna, ine17, ine12, ine02)
    named <- merge(ref, clone, by = 'ine17')

  } else if (level == 'zc') {

    names(clone)[names(clone) == stringr::str_glue('{common_var}')] <- 'zona'
    clone <- clone |>
      dplyr::mutate(
        zona = stringr::str_pad(zona, 11, 'left', '0'),
        ine17 = stringr::str_sub(zona, 1, 5)
      )
    ref <- ref |> dplyr::select(region, id_region, provincia, id_provin, comuna, ine17, ine12, ine02)
    named <- merge(ref, clone, by = 'ine17')

  }

  return(named)

}
