/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Tconv.h"    
#include "H5Tconv_macros.h"
#include "H5Tconv_complex.h"
#include "H5Tconv_integer.h"
#include "H5Tconv_float.h"

static herr_t H5T__conv_complex_loop(const H5T_t *src_p, const H5T_t *dst_p, const H5T_conv_ctx_t *conv_ctx,
                                     size_t nelmts, size_t buf_stride, void *buf);
static herr_t H5T__conv_complex_part(const H5T_t *src_p, const H5T_t *dst_p, uint8_t *s, uint8_t *d,
                                     const H5T_conv_ctx_t *conv_ctx, uint8_t *src_rev,
                                     bool *exception_handled);

herr_t
H5T__conv_complex(const H5T_t *src_p, const H5T_t *dst_p, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                  size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                  void H5_ATTR_UNUSED *bkg)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            H5T_atomic_t src_atomic; 
            H5T_atomic_t dst_atomic; 

            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (!H5T_IS_ATOMIC(src_p->shared->parent->shared))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid source complex number datatype");
            if (!H5T_IS_ATOMIC(dst_p->shared->parent->shared))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid destination complex number datatype");
            src_atomic = src_p->shared->parent->shared->u.atomic;
            dst_atomic = dst_p->shared->parent->shared->u.atomic;
            if (H5T_ORDER_LE != src_atomic.order && H5T_ORDER_BE != src_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for source datatype");
            if (H5T_ORDER_LE != dst_atomic.order && H5T_ORDER_BE != dst_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for destination datatype");
            if (dst_p->shared->size > 2 * TEMP_FLOAT_CONV_BUFFER_SIZE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "destination datatype size is too large");
            if (8 * sizeof(int64_t) - 1 < src_atomic.u.f.esize ||
                8 * sizeof(int64_t) - 1 < dst_atomic.u.f.esize)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "exponent field is too large");
            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");

            if (H5T__conv_complex_loop(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");

            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__conv_complex_loop(const H5T_t *src_p, const H5T_t *dst_p, const H5T_conv_ctx_t *conv_ctx, size_t nelmts,
                       size_t buf_stride, void *buf)
{
    H5T_conv_float_specval_t realval_type; 
    H5T_conv_float_specval_t imagval_type; 
    H5T_conv_ret_t           except_ret;   
    H5T_atomic_t             src_atomic;   
    H5T_atomic_t             dst_atomic;   
    ssize_t  src_delta, dst_delta;         
    uint8_t *s, *sp, *d, *dp;              
    uint8_t *src_rev = NULL;               
    uint8_t  dbuf[2 * TEMP_FLOAT_CONV_BUFFER_SIZE]; 
    size_t   src_part_size; 
    size_t   dst_part_size; 
    size_t   olap;          
    int      direction;     
    herr_t   ret_value = SUCCEED;

    assert(src_p);
    assert(src_p->shared->type == H5T_COMPLEX);
    assert(dst_p);
    assert(dst_p->shared->type == H5T_COMPLEX);
    assert(conv_ctx);
    assert(buf);

    FUNC_ENTER_PACKAGE

    src_atomic    = src_p->shared->parent->shared->u.atomic;
    dst_atomic    = dst_p->shared->parent->shared->u.atomic;
    src_part_size = src_p->shared->size / 2;
    dst_part_size = dst_p->shared->size / 2;

    
    if (src_p->shared->size == dst_p->shared->size || buf_stride) {
        sp = dp   = (uint8_t *)buf;
        direction = 1;
        olap      = nelmts;
    }
    else if (src_p->shared->size >= dst_p->shared->size) {
        double olap_d =
            ceil((double)(dst_p->shared->size) / (double)(src_p->shared->size - dst_p->shared->size));
        olap = (size_t)olap_d;
        sp = dp   = (uint8_t *)buf;
        direction = 1;
    }
    else {
        double olap_d =
            ceil((double)(src_p->shared->size) / (double)(dst_p->shared->size - src_p->shared->size));
        olap      = (size_t)olap_d;
        sp        = (uint8_t *)buf + (nelmts - 1) * src_p->shared->size;
        dp        = (uint8_t *)buf + (nelmts - 1) * dst_p->shared->size;
        direction = -1;
    }

    
    H5_CHECK_OVERFLOW(buf_stride, size_t, ssize_t);
    H5_CHECK_OVERFLOW(src_p->shared->size, size_t, ssize_t);
    H5_CHECK_OVERFLOW(dst_p->shared->size, size_t, ssize_t);
    src_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : src_p->shared->size);
    dst_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : dst_p->shared->size);

    
    if (conv_ctx->u.conv.cb_struct.func)
        if (NULL == (src_rev = H5MM_calloc(src_p->shared->size)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "couldn't allocate temporary buffer");

    
    for (size_t elmtno = 0; elmtno < nelmts; elmtno++) {
        bool reverse     = true;  
        bool real_zero   = false; 
        bool imag_zero   = false; 
        bool real_except = false; 
        bool imag_except = false; 

        
        s = sp;
        if (direction > 0)
            d = elmtno < olap ? dbuf : dp;
        else
            d = elmtno + olap >= nelmts ? dbuf : dp;
        if (d == dbuf)
            memset(dbuf, 0, sizeof(dbuf));

#ifndef NDEBUG
        if (d == dbuf) {
            assert((dp >= sp && dp < sp + src_p->shared->size) ||
                   (sp >= dp && sp < dp + dst_p->shared->size));
        }
        else {
            assert((dp < sp && dp + dst_p->shared->size <= sp) ||
                   (sp < dp && sp + src_p->shared->size <= dp));
        }
#endif

        
        if (H5T_ORDER_BE == src_atomic.order) {
            uint8_t *cur_part = s;
            
            for (size_t j = 0; j < src_part_size / 2; j++)
                H5_SWAP_BYTES(cur_part, j, src_part_size - (j + 1));
            
            cur_part += src_part_size;
            for (size_t j = 0; j < src_part_size / 2; j++)
                H5_SWAP_BYTES(cur_part, j, src_part_size - (j + 1));
        }
        else if (H5T_ORDER_VAX == src_atomic.order)
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                        "VAX byte ordering is unsupported for complex number type conversions");

        
        realval_type = H5T__conv_float_find_special(s, &src_atomic, NULL);
        imagval_type = H5T__conv_float_find_special(s + (src_p->shared->size / 2), &src_atomic, NULL);

        real_zero = (realval_type == H5T_CONV_FLOAT_SPECVAL_POSZERO ||
                     realval_type == H5T_CONV_FLOAT_SPECVAL_NEGZERO);
        imag_zero = (imagval_type == H5T_CONV_FLOAT_SPECVAL_POSZERO ||
                     imagval_type == H5T_CONV_FLOAT_SPECVAL_NEGZERO);
        real_except =
            (realval_type == H5T_CONV_FLOAT_SPECVAL_POSINF || realval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF ||
             realval_type == H5T_CONV_FLOAT_SPECVAL_NAN);
        imag_except =
            (imagval_type == H5T_CONV_FLOAT_SPECVAL_POSINF || imagval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF ||
             imagval_type == H5T_CONV_FLOAT_SPECVAL_NAN);

        
        if (real_zero && imag_zero) {
            H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
            H5T__bit_copy(d + dst_part_size, dst_atomic.u.f.sign, s + src_part_size, src_atomic.u.f.sign,
                          (size_t)1);
            H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, false);
            H5T__bit_set(d + dst_part_size, dst_atomic.u.f.epos, dst_atomic.u.f.esize, false);
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            H5T__bit_set(d + dst_part_size, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            goto padding;
        }
        else if (real_except || imag_except) {
            except_ret = H5T_CONV_UNHANDLED;

            
            if (conv_ctx->u.conv.cb_struct.func) {
                H5T_conv_except_t except_type; 

                
                H5T__reverse_order(src_rev, s, src_p);

                
                if (realval_type == H5T_CONV_FLOAT_SPECVAL_POSINF)
                    except_type = H5T_CONV_EXCEPT_PINF;
                else if (realval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF)
                    except_type = H5T_CONV_EXCEPT_NINF;
                else if (imagval_type == H5T_CONV_FLOAT_SPECVAL_POSINF)
                    except_type = H5T_CONV_EXCEPT_PINF;
                else if (imagval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF)
                    except_type = H5T_CONV_EXCEPT_NINF;
                else {
                    assert(realval_type == H5T_CONV_FLOAT_SPECVAL_NAN ||
                           imagval_type == H5T_CONV_FLOAT_SPECVAL_NAN);
                    except_type = H5T_CONV_EXCEPT_NAN;
                }

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        except_ret = (conv_ctx->u.conv.cb_struct.func)(
                            except_type, conv_ctx->u.conv.src_type_id, conv_ctx->u.conv.dst_type_id, src_rev,
                            d, conv_ctx->u.conv.cb_struct.user_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            }

            if (except_ret == H5T_CONV_UNHANDLED) {
                if (realval_type == H5T_CONV_FLOAT_SPECVAL_POSINF ||
                    realval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF ||
                    imagval_type == H5T_CONV_FLOAT_SPECVAL_POSINF ||
                    imagval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF) {
                    H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
                    H5T__bit_copy(d + dst_part_size, dst_atomic.u.f.sign, s + src_part_size,
                                  src_atomic.u.f.sign, (size_t)1);

                    if (realval_type == H5T_CONV_FLOAT_SPECVAL_POSINF ||
                        realval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF) {
                        H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, true);
                        H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
                        
                        if (H5T_NORM_NONE == dst_atomic.u.f.norm)
                            H5T__bit_set(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - 1, (size_t)1, true);
                    }
                    if (imagval_type == H5T_CONV_FLOAT_SPECVAL_POSINF ||
                        imagval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF) {
                        H5T__bit_set(d + dst_part_size, dst_atomic.u.f.epos, dst_atomic.u.f.esize, true);
                        H5T__bit_set(d + dst_part_size, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
                        
                        if (H5T_NORM_NONE == dst_atomic.u.f.norm)
                            H5T__bit_set(d + dst_part_size, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - 1,
                                         (size_t)1, true);
                    }
                }
                else {
                    
                    if (realval_type == H5T_CONV_FLOAT_SPECVAL_NAN) {
                        H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
                        H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, true);
                        H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, true);
                    }
                    if (imagval_type == H5T_CONV_FLOAT_SPECVAL_NAN) {
                        H5T__bit_copy(d + dst_part_size, dst_atomic.u.f.sign, s + src_part_size,
                                      src_atomic.u.f.sign, (size_t)1);
                        H5T__bit_set(d + dst_part_size, dst_atomic.u.f.epos, dst_atomic.u.f.esize, true);
                        H5T__bit_set(d + dst_part_size, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, true);
                    }
                }
            }
            else if (except_ret == H5T_CONV_HANDLED) {
                
                reverse = false;
                goto next;
            }
            else if (except_ret == H5T_CONV_ABORT)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");

            goto padding;
        }

        if (real_zero) {
            H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
            H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, false);
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
        }
        else {
            bool exception_handled = false;

            if (H5T__conv_complex_part(src_p, dst_p, s, d, conv_ctx, src_rev, &exception_handled) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't convert real part of complex number");

            
            if (exception_handled) {
                reverse = false;
                goto next;
            }
        }

        if (imag_zero) {
            H5T__bit_copy(d + dst_part_size, dst_atomic.u.f.sign, s + src_part_size, src_atomic.u.f.sign,
                          (size_t)1);
            H5T__bit_set(d + dst_part_size, dst_atomic.u.f.epos, dst_atomic.u.f.esize, false);
            H5T__bit_set(d + dst_part_size, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
        }
        else {
            bool exception_handled = false;

            if (H5T__conv_complex_part(src_p, dst_p, s + src_part_size, d + dst_part_size, conv_ctx, src_rev,
                                       &exception_handled) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL,
                            "can't convert imaginary part of complex number");

            
            if (exception_handled) {
                reverse = false;
                goto next;
            }
        }

padding:
        
        if (dst_atomic.offset > 0) {
            assert(H5T_PAD_ZERO == dst_atomic.lsb_pad || H5T_PAD_ONE == dst_atomic.lsb_pad);
            H5T__bit_set(d, (size_t)0, dst_atomic.offset, (bool)(H5T_PAD_ONE == dst_atomic.lsb_pad));
            H5T__bit_set(d + dst_part_size, (size_t)0, dst_atomic.offset,
                         (bool)(H5T_PAD_ONE == dst_atomic.lsb_pad));
        }
        {
            size_t type_size = dst_p->shared->parent->shared->size;

            if (dst_atomic.offset + dst_atomic.prec != 8 * type_size) {
                assert(H5T_PAD_ZERO == dst_atomic.msb_pad || H5T_PAD_ONE == dst_atomic.msb_pad);
                H5T__bit_set(d, dst_atomic.offset + dst_atomic.prec,
                             8 * type_size - (dst_atomic.offset + dst_atomic.prec),
                             (bool)(H5T_PAD_ONE == dst_atomic.msb_pad));
                H5T__bit_set(d + dst_part_size, dst_atomic.offset + dst_atomic.prec,
                             8 * type_size - (dst_atomic.offset + dst_atomic.prec),
                             (bool)(H5T_PAD_ONE == dst_atomic.msb_pad));
            }
        }

        
        if (H5T_ORDER_BE == dst_atomic.order && reverse) {
            uint8_t *cur_part = d;
            
            for (size_t j = 0; j < dst_part_size / 2; j++)
                H5_SWAP_BYTES(cur_part, j, dst_part_size - (j + 1));
            
            cur_part += dst_part_size;
            for (size_t j = 0; j < dst_part_size / 2; j++)
                H5_SWAP_BYTES(cur_part, j, dst_part_size - (j + 1));
        }
        else if (H5T_ORDER_VAX == dst_atomic.order)
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                        "VAX byte ordering is unsupported for complex number type conversions");

next:
        
        if (d == dbuf)
            H5MM_memcpy(dp, d, dst_p->shared->size);

        
        sp += src_delta;
        dp += dst_delta;
    } 

done:
    H5MM_free(src_rev);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__conv_complex_part(const H5T_t *src_p, const H5T_t *dst_p, uint8_t *s, uint8_t *d,
                       const H5T_conv_ctx_t *conv_ctx, uint8_t *src_rev, bool *exception_handled)
{
    H5T_conv_ret_t except_ret = H5T_CONV_UNHANDLED; 
    H5T_atomic_t   src_atomic;                      
    H5T_atomic_t   dst_atomic;                      
    hssize_t       expo_max;                        
    ssize_t        mant_msb = 0;                    
    int64_t        expo;                            
    size_t         msize = 0;                       
    size_t         mpos;                            
    size_t         mrsh;                            
    size_t         implied;                         
    bool           denormalized = false;            
    bool           carry        = false;            
    herr_t         ret_value    = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(src_p);
    assert(dst_p);
    assert(s);
    assert(d);
    assert(conv_ctx);
    assert(exception_handled);

    if (conv_ctx->u.conv.cb_struct.func)
        assert(src_rev);

    *exception_handled = false;

    src_atomic = src_p->shared->parent->shared->u.atomic;
    dst_atomic = dst_p->shared->parent->shared->u.atomic;
    expo_max   = ((hssize_t)1 << dst_atomic.u.f.esize) - 1;

    
    expo = (int64_t)H5T__bit_get_d(s, src_atomic.u.f.epos, src_atomic.u.f.esize);

    if (expo == 0)
        denormalized = true;

    
    if (0 == expo || H5T_NORM_NONE == src_atomic.u.f.norm) {
        if ((mant_msb = H5T__bit_find(s, src_atomic.u.f.mpos, src_atomic.u.f.msize, H5T_BIT_MSB, true)) > 0)
            msize = (size_t)mant_msb;
        else if (0 == mant_msb) {
            msize = 1;
            H5T__bit_set(s, src_atomic.u.f.mpos, (size_t)1, false);
        }
    }
    else if (H5T_NORM_IMPLIED == src_atomic.u.f.norm)
        msize = src_atomic.u.f.msize;
    else
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "normalization method not implemented yet");

    
    H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);

    
    if (0 == expo || H5T_NORM_NONE == src_atomic.u.f.norm) {
        assert(mant_msb >= 0);
        expo -= (int64_t)((src_atomic.u.f.ebias - 1) + (src_atomic.u.f.msize - (size_t)mant_msb));
    }
    else if (H5T_NORM_IMPLIED == src_atomic.u.f.norm)
        expo -= (int64_t)src_atomic.u.f.ebias;
    else
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "normalization method not implemented yet");

    
    mrsh = 0;
    if (H5T_NORM_NONE == dst_atomic.u.f.norm)
        mrsh++;

    
    expo += (int64_t)dst_atomic.u.f.ebias;

    if (expo < -(hssize_t)(dst_atomic.u.f.msize)) {
        
        expo = 0;
        H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
        msize = 0;
    }
    else if (expo <= 0) {
        
        mrsh += (size_t)(1 - expo);
        expo         = 0;
        denormalized = true;
    }
    else if (expo >= expo_max) {
        
        if (conv_ctx->u.conv.cb_struct.func) { 
            
            H5T__reverse_order(src_rev, s, src_p);

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    except_ret = (conv_ctx->u.conv.cb_struct.func)(
                        H5T_CONV_EXCEPT_RANGE_HI, conv_ctx->u.conv.src_type_id, conv_ctx->u.conv.dst_type_id,
                        src_rev, d, conv_ctx->u.conv.cb_struct.user_data);
                }
            H5_AFTER_USER_CB(FAIL)
        }

        if (except_ret == H5T_CONV_UNHANDLED) {
            expo = expo_max;
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            msize = 0;
        }
        else if (except_ret == H5T_CONV_HANDLED) {
            *exception_handled = true;
            goto done;
        }
        else if (except_ret == H5T_CONV_ABORT)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
    }

    
    implied = 1;
    mpos    = src_atomic.u.f.mpos;
    if (msize > 0 && mrsh <= dst_atomic.u.f.msize && mrsh + msize > dst_atomic.u.f.msize) {
        mant_msb = (ssize_t)(mrsh + msize - dst_atomic.u.f.msize);
        assert(mant_msb >= 0 && (size_t)mant_msb <= msize);
        
        if (H5T__bit_get_d(s, (mpos + (size_t)mant_msb) - 1, (size_t)1) && !denormalized) {
            
            if ((H5T__bit_find(s, mpos + (size_t)mant_msb, msize - (size_t)mant_msb, H5T_BIT_LSB, false) >=
                     0 ||
                 expo < expo_max - 1)) {
                carry = H5T__bit_inc(s, mpos + (size_t)mant_msb - 1, 1 + msize - (size_t)mant_msb);
                if (carry)
                    implied = 2;
            }
        }
        else if (H5T__bit_get_d(s, (mpos + (size_t)mant_msb) - 1, (size_t)1) && denormalized)
            
            H5T__bit_inc(s, mpos + (size_t)mant_msb - 1, 1 + msize - (size_t)mant_msb);
    }
    else
        carry = false;

    
    if (mrsh > dst_atomic.u.f.msize + 1) {
        H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
    }
    else if (mrsh == dst_atomic.u.f.msize + 1) {
        H5T__bit_set(d, dst_atomic.u.f.mpos + 1, dst_atomic.u.f.msize - 1, false);
        H5T__bit_set(d, dst_atomic.u.f.mpos, (size_t)1, true);
    }
    else if (mrsh == dst_atomic.u.f.msize) {
        H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
        H5T__bit_set_d(d, dst_atomic.u.f.mpos, MIN(2, dst_atomic.u.f.msize), (hsize_t)implied);
    }
    else {
        if (mrsh > 0) {
            H5T__bit_set(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - mrsh, mrsh, false);
            H5T__bit_set_d(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - mrsh, (size_t)2, (hsize_t)implied);
        }
        if (mrsh + msize >= dst_atomic.u.f.msize) {
            H5T__bit_copy(d, dst_atomic.u.f.mpos, s, (mpos + msize + mrsh - dst_atomic.u.f.msize),
                          dst_atomic.u.f.msize - mrsh);
        }
        else {
            H5T__bit_copy(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - (mrsh + msize), s, mpos, msize);
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize - (mrsh + msize), false);
        }
    }

    
    if (carry) {
        expo++;
        if (expo >= expo_max) {
            
            if (conv_ctx->u.conv.cb_struct.func) { 
                
                H5T__reverse_order(src_rev, s, src_p);

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        except_ret = (conv_ctx->u.conv.cb_struct.func)(
                            H5T_CONV_EXCEPT_RANGE_HI, conv_ctx->u.conv.src_type_id,
                            conv_ctx->u.conv.dst_type_id, src_rev, d, conv_ctx->u.conv.cb_struct.user_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            }

            if (except_ret == H5T_CONV_UNHANDLED) {
                expo = expo_max;
                H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            }
            else if (except_ret == H5T_CONV_HANDLED) {
                *exception_handled = true;
                goto done;
            }
            else if (except_ret == H5T_CONV_ABORT)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");
        }
    }

    H5_CHECK_OVERFLOW(expo, hssize_t, hsize_t);
    H5T__bit_set_d(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, (hsize_t)expo);

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

herr_t
H5T__conv_complex_i(const H5T_t *src_p, const H5T_t *dst_p, H5T_cdata_t *cdata,
                    const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                    size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            H5T_atomic_t src_atomic; 
            H5T_atomic_t dst_atomic; 

            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (!H5T_IS_ATOMIC(src_p->shared->parent->shared))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid complex number datatype");
            src_atomic = src_p->shared->parent->shared->u.atomic;
            dst_atomic = dst_p->shared->u.atomic;
            if (H5T_ORDER_LE != src_atomic.order && H5T_ORDER_BE != src_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for source datatype");
            if (H5T_ORDER_LE != dst_atomic.order && H5T_ORDER_BE != dst_atomic.order &&
                H5T_ORDER_VAX != dst_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for destination datatype");
            if (dst_p->shared->size > TEMP_INT_CONV_BUFFER_SIZE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "destination datatype size is too large");
            if (8 * sizeof(hssize_t) - 1 < src_atomic.u.f.esize)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "exponent field is too large");
            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");

            if (H5T__conv_f_i_loop(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");

            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_complex_f(const H5T_t *src_p, const H5T_t *dst_p, H5T_cdata_t *cdata,
                    const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                    size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    bool   equal_cplx_conv = false; 
    herr_t ret_value       = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            H5T_atomic_t src_atomic; 
            H5T_atomic_t dst_atomic; 

            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (!H5T_IS_ATOMIC(src_p->shared->parent->shared))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid complex number datatype");
            src_atomic = src_p->shared->parent->shared->u.atomic;
            dst_atomic = dst_p->shared->u.atomic;
            if (H5T_ORDER_LE != src_atomic.order && H5T_ORDER_BE != src_atomic.order &&
                H5T_ORDER_VAX != src_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for source datatype");
            if (H5T_ORDER_LE != dst_atomic.order && H5T_ORDER_BE != dst_atomic.order &&
                H5T_ORDER_VAX != dst_atomic.order)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported byte order for destination datatype");
            if (dst_p->shared->size > TEMP_FLOAT_CONV_BUFFER_SIZE)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "destination datatype size is too large");
            if (8 * sizeof(int64_t) - 1 < src_atomic.u.f.esize ||
                8 * sizeof(int64_t) - 1 < dst_atomic.u.f.esize)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "exponent field is too large");
            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            if (!src_p || !dst_p)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a datatype");
            if (NULL == conv_ctx)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid datatype conversion context pointer");

            
            equal_cplx_conv = (0 == H5T_cmp(src_p->shared->parent, dst_p, false));
            if (!equal_cplx_conv) {
                
                if (H5T__conv_f_f_loop(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");
            }
            else {
                
                if (H5T__conv_complex_f_matched(src_p, dst_p, conv_ctx, nelmts, buf_stride, buf) < 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "unable to convert data values");
            }

            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_complex_f_matched(const H5T_t *src_p, const H5T_t *dst_p, const H5T_conv_ctx_t *conv_ctx,
                            size_t nelmts, size_t buf_stride, void *buf)
{
    H5T_conv_float_specval_t specval_type;      
    H5T_atomic_t             src_atomic;        
    H5T_atomic_t             dst_atomic;        
    ssize_t  src_delta, dst_delta;              
    uint8_t *s, *sp, *d, *dp;                   
    uint8_t *src_rev = NULL;                    
    uint8_t  dbuf[TEMP_FLOAT_CONV_BUFFER_SIZE]; 
    size_t   olap;                              
    int      direction;                         
    herr_t   ret_value = SUCCEED;

    assert(src_p);
    assert(src_p->shared->type == H5T_FLOAT || src_p->shared->type == H5T_COMPLEX);
    assert(dst_p);
    assert(dst_p->shared->type == H5T_FLOAT || dst_p->shared->type == H5T_COMPLEX);
    assert(conv_ctx);
    assert(buf);

    FUNC_ENTER_PACKAGE

    if (src_p->shared->type == H5T_COMPLEX)
        src_atomic = src_p->shared->parent->shared->u.atomic;
    else
        src_atomic = src_p->shared->u.atomic;
    if (dst_p->shared->type == H5T_COMPLEX)
        dst_atomic = dst_p->shared->parent->shared->u.atomic;
    else
        dst_atomic = dst_p->shared->u.atomic;

#ifndef NDEBUG
    {
        
        const H5T_t *src_base = (src_p->shared->type == H5T_FLOAT) ? src_p : src_p->shared->parent;
        const H5T_t *dst_base = (dst_p->shared->type == H5T_FLOAT) ? dst_p : dst_p->shared->parent;
        assert(0 == (H5T_cmp(src_base, dst_base, false)));
    }
#endif

    
    if (src_p->shared->size == dst_p->shared->size || buf_stride) {
        sp = dp   = (uint8_t *)buf;
        direction = 1;
        olap      = nelmts;
    }
    else if (src_p->shared->size >= dst_p->shared->size) {
        double olap_d =
            ceil((double)(dst_p->shared->size) / (double)(src_p->shared->size - dst_p->shared->size));
        olap = (size_t)olap_d;
        sp = dp   = (uint8_t *)buf;
        direction = 1;
    }
    else {
        double olap_d =
            ceil((double)(src_p->shared->size) / (double)(dst_p->shared->size - src_p->shared->size));
        olap      = (size_t)olap_d;
        sp        = (uint8_t *)buf + (nelmts - 1) * src_p->shared->size;
        dp        = (uint8_t *)buf + (nelmts - 1) * dst_p->shared->size;
        direction = -1;
    }

    
    H5_CHECK_OVERFLOW(buf_stride, size_t, ssize_t);
    H5_CHECK_OVERFLOW(src_p->shared->size, size_t, ssize_t);
    H5_CHECK_OVERFLOW(dst_p->shared->size, size_t, ssize_t);
    src_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : src_p->shared->size);
    dst_delta = (ssize_t)direction * (ssize_t)(buf_stride ? buf_stride : dst_p->shared->size);

    
    if (conv_ctx->u.conv.cb_struct.func)
        if (NULL == (src_rev = H5MM_calloc(src_p->shared->size)))
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTALLOC, FAIL, "couldn't allocate temporary buffer");

    
    for (size_t elmtno = 0; elmtno < nelmts; elmtno++) {
        H5T_conv_ret_t except_ret = H5T_CONV_UNHANDLED; 
        bool           reverse    = true;               

        
        s = sp;
        if (direction > 0)
            d = elmtno < olap ? dbuf : dp;
        else
            d = elmtno + olap >= nelmts ? dbuf : dp;
        if (d == dbuf)
            memset(dbuf, 0, sizeof(dbuf));

#ifndef NDEBUG
        if (d == dbuf) {
            assert((dp >= sp && dp < sp + src_p->shared->size) ||
                   (sp >= dp && sp < dp + dst_p->shared->size));
        }
        else {
            assert((dp < sp && dp + dst_p->shared->size <= sp) ||
                   (sp < dp && sp + src_p->shared->size <= dp));
        }
#endif

        
        if (H5T_ORDER_BE == src_atomic.order) {
            size_t half_size = src_p->shared->size / 2;

            if (H5T_FLOAT == src_p->shared->type) {
                for (size_t j = 0; j < half_size; j++)
                    H5_SWAP_BYTES(s, j, src_p->shared->size - (j + 1));
            }
            else {
                uint8_t *cur_part = s;
                
                for (size_t j = 0; j < half_size / 2; j++)
                    H5_SWAP_BYTES(cur_part, j, half_size - (j + 1));
                
                cur_part += half_size;
                for (size_t j = 0; j < half_size / 2; j++)
                    H5_SWAP_BYTES(cur_part, j, half_size - (j + 1));
            }
        }
        else if (H5T_ORDER_VAX == src_atomic.order) {
            if (H5T_FLOAT == src_p->shared->type) {
                uint8_t tmp1, tmp2;
                size_t  tsize = src_p->shared->size;
                assert(0 == tsize % 2);

                for (size_t i = 0; i < tsize; i += 4) {
                    tmp1 = s[i];
                    tmp2 = s[i + 1];

                    s[i]     = s[(tsize - 2) - i];
                    s[i + 1] = s[(tsize - 1) - i];

                    s[(tsize - 2) - i] = tmp1;
                    s[(tsize - 1) - i] = tmp2;
                }
            }
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "VAX byte ordering is unsupported for complex number type conversions");
        }

        
        specval_type = H5T__conv_float_find_special(s, &src_atomic, NULL);
        if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSZERO ||
            specval_type == H5T_CONV_FLOAT_SPECVAL_NEGZERO) {
            H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
            H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, false);
            H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
            goto padding;
        }
        else if (specval_type != H5T_CONV_FLOAT_SPECVAL_REGULAR) {
            
            if (conv_ctx->u.conv.cb_struct.func) {
                H5T_conv_except_t except_type; 

                
                H5T__reverse_order(src_rev, s, src_p);

                if (specval_type == H5T_CONV_FLOAT_SPECVAL_POSINF)
                    except_type = H5T_CONV_EXCEPT_PINF;
                else if (specval_type == H5T_CONV_FLOAT_SPECVAL_NEGINF)
                    except_type = H5T_CONV_EXCEPT_NINF;
                else
                    except_type = H5T_CONV_EXCEPT_NAN;

                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        except_ret = (conv_ctx->u.conv.cb_struct.func)(
                            except_type, conv_ctx->u.conv.src_type_id, conv_ctx->u.conv.dst_type_id, src_rev,
                            d, conv_ctx->u.conv.cb_struct.user_data);
                    }
                H5_AFTER_USER_CB(FAIL)
            }

            if (except_ret == H5T_CONV_UNHANDLED) {
                H5T__bit_copy(d, dst_atomic.u.f.sign, s, src_atomic.u.f.sign, (size_t)1);
                H5T__bit_set(d, dst_atomic.u.f.epos, dst_atomic.u.f.esize, true);
                if (specval_type == H5T_CONV_FLOAT_SPECVAL_NAN)
                    
                    H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, true);
                else {
                    
                    H5T__bit_set(d, dst_atomic.u.f.mpos, dst_atomic.u.f.msize, false);
                    
                    if (H5T_NORM_NONE == dst_atomic.u.f.norm)
                        H5T__bit_set(d, dst_atomic.u.f.mpos + dst_atomic.u.f.msize - 1, (size_t)1, true);
                }
            }
            else if (except_ret == H5T_CONV_HANDLED) {
                
                reverse = false;
                goto next;
            }
            else if (except_ret == H5T_CONV_ABORT)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCONVERT, FAIL, "can't handle conversion exception");

            goto padding;
        }

        
        if (H5T_FLOAT == src_p->shared->type)
            memcpy(d, s, src_p->shared->size);
        else
            memcpy(d, s, src_p->shared->size / 2);

padding:
        
        if (dst_atomic.offset > 0) {
            assert(H5T_PAD_ZERO == dst_atomic.lsb_pad || H5T_PAD_ONE == dst_atomic.lsb_pad);
            H5T__bit_set(d, (size_t)0, dst_atomic.offset, (bool)(H5T_PAD_ONE == dst_atomic.lsb_pad));
        }
        {
            size_t type_size;

            if (dst_p->shared->type == H5T_FLOAT)
                type_size = dst_p->shared->size;
            else
                type_size = dst_p->shared->parent->shared->size;

            if (dst_atomic.offset + dst_atomic.prec != 8 * type_size) {
                assert(H5T_PAD_ZERO == dst_atomic.msb_pad || H5T_PAD_ONE == dst_atomic.msb_pad);
                H5T__bit_set(d, dst_atomic.offset + dst_atomic.prec,
                             8 * type_size - (dst_atomic.offset + dst_atomic.prec),
                             (bool)(H5T_PAD_ONE == dst_atomic.msb_pad));
            }
        }

        
        if (H5T_ORDER_BE == dst_atomic.order && reverse) {
            size_t half_size = dst_p->shared->size / 2;

            if (H5T_FLOAT == dst_p->shared->type) {
                for (size_t j = 0; j < half_size; j++)
                    H5_SWAP_BYTES(d, j, dst_p->shared->size - (j + 1));
            }
            else {
                for (size_t j = 0; j < half_size / 2; j++)
                    H5_SWAP_BYTES(d, j, half_size - (j + 1));
            }
        }
        else if (H5T_ORDER_VAX == dst_atomic.order && reverse) {
            if (H5T_FLOAT == dst_p->shared->type) {
                uint8_t tmp1, tmp2;
                size_t  tsize = dst_p->shared->size / 2;
                assert(0 == tsize % 2);

                for (size_t i = 0; i < tsize; i += 4) {
                    tmp1 = d[i];
                    tmp2 = d[i + 1];

                    d[i]     = d[(tsize - 2) - i];
                    d[i + 1] = d[(tsize - 1) - i];

                    d[(tsize - 2) - i] = tmp1;
                    d[(tsize - 1) - i] = tmp2;
                }
            }
            else
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "VAX byte ordering is unsupported for complex number type conversions");
        }

next:
        
        if (d == dbuf) {
            if (H5T_FLOAT == dst_p->shared->type)
                H5MM_memcpy(dp, d, dst_p->shared->size);
            else
                H5MM_memcpy(dp, d, dst_p->shared->size / 2);
        }

        
        if (H5T_COMPLEX == dst_p->shared->type)
            memset(dp + (dst_p->shared->size / 2), 0, dst_p->shared->size / 2);

        
        sp += src_delta;
        dp += dst_delta;
    } 

done:
    H5MM_free(src_rev);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5T__conv_complex_compat(const H5T_t *src, const H5T_t *dst, H5T_cdata_t *cdata,
                         const H5T_conv_ctx_t H5_ATTR_UNUSED *conv_ctx, size_t H5_ATTR_UNUSED nelmts,
                         size_t H5_ATTR_UNUSED buf_stride, size_t H5_ATTR_UNUSED bkg_stride,
                         void H5_ATTR_UNUSED *_buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_t *compound_copy = NULL;
    herr_t ret_value     = SUCCEED;

    FUNC_ENTER_PACKAGE

    switch (cdata->command) {
        case H5T_CONV_INIT: {
            const H5T_t *complex_type;
            const H5T_t *other_type;

            if (src->shared->type == H5T_COMPLEX) {
                if (dst->shared->type != H5T_ARRAY && dst->shared->type != H5T_COMPOUND)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "unsupported destination datatype for conversion");
                complex_type = src;
                other_type   = dst;
            }
            else {
                if (dst->shared->type != H5T_COMPLEX)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "unsupported destination datatype for conversion");
                complex_type = dst;
                other_type   = src;
            }

            if (complex_type->shared->u.cplx.form != H5T_COMPLEX_RECTANGULAR)
                HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                            "unsupported form of complex number datatype for conversion");

            if (other_type->shared->type == H5T_ARRAY) {
                if (other_type->shared->u.array.nelem != 2)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "array datatype doesn't have the correct number of elements for conversion");
                if (H5T_cmp(other_type->shared->parent, complex_type->shared->parent, false))
                    HGOTO_ERROR(
                        H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                        "array datatype's base datatype doesn't match complex number type's base datatype");
            }
            else {
                H5T_cmemb_t *fields;
                size_t       name_len;

                assert(other_type->shared->type == H5T_COMPOUND);

                if (other_type->shared->u.compnd.nmembs != 2)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype doesn't have the correct number of fields for conversion");
                if (!other_type->shared->u.compnd.packed)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype fields aren't packed together");
                if (other_type->shared->u.compnd.memb_size != complex_type->shared->size)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype size doesn't match size of complex number datatype");

                
                if (other_type->shared->u.compnd.sorted == H5T_SORT_NONE ||
                    other_type->shared->u.compnd.sorted == H5T_SORT_VALUE)
                    fields = other_type->shared->u.compnd.memb;
                else {
                    
                    if (NULL == (compound_copy = H5T_copy(other_type, H5T_COPY_TRANSIENT)))
                        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCOPY, FAIL, "can't copy datatype");

                    H5T__sort_value(compound_copy, NULL);
                    fields = compound_copy->shared->u.compnd.memb;
                }

                
                if (fields[0].offset != 0)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'real' field is not at offset 0");
                if (fields[0].size != complex_type->shared->parent->shared->size)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'real' field is not the same size as the complex number "
                                "datatype's base datatype");

                
                name_len = strlen(fields[0].name);
                if (strncmp(fields[0].name, "real", (name_len < 5) ? name_len : 5))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'real' field name ('%s') didn't match an expected name "
                                "for conversion",
                                fields[0].name);

                if (H5T_cmp(fields[0].type, complex_type->shared->parent, false))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'real' field is not the same datatype as the complex "
                                "number datatype's base datatype");

                
                if (fields[1].offset != fields[0].size)
                    HGOTO_ERROR(
                        H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                        "compound datatype's 'imaginary' field is not at offset 'sizeof(real_field)'");
                if (fields[1].size != complex_type->shared->parent->shared->size)
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'imaginary' field is not the same size as the complex "
                                "number datatype's base datatype");

                
                name_len = strlen(fields[1].name);
                if (strncmp(fields[1].name, "imaginary", (name_len < 10) ? name_len : 10))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'imaginary' field name ('%s') didn't match an expected "
                                "name for conversion",
                                fields[1].name);

                if (H5T_cmp(fields[1].type, complex_type->shared->parent, false))
                    HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL,
                                "compound datatype's 'imaginary' field is not the same datatype as the "
                                "complex number datatype's base datatype");
            }

            cdata->need_bkg = H5T_BKG_NO;

            break;
        }

        case H5T_CONV_FREE:
            break;

        case H5T_CONV_CONV:
            
            break;

        default:
            HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "unknown conversion command");
    }

done:
    if (compound_copy && H5T_close(compound_copy) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "can't close datatype");

    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_COMPLEX_NUMBERS

herr_t
H5T__conv_fcomplex_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, SCHAR, H5_float_complex, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, UCHAR, H5_float_complex, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, SHORT, H5_float_complex, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, USHORT, H5_float_complex, unsigned short, 0, USHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, INT, H5_float_complex, int, INT_MIN, INT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, UINT, H5_float_complex, unsigned int, 0, UINT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, LONG, H5_float_complex, long, LONG_MIN, LONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, ULONG, H5_float_complex, unsigned long, 0, ULONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, LLONG, H5_float_complex, long long, LLONG_MIN, LLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_fcomplex_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(FLOAT_COMPLEX, ULLONG, H5_float_complex, unsigned long long, 0, ULLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

#ifdef H5_HAVE__FLOAT16

herr_t
H5T__conv_fcomplex__Float16(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    
    H5_WARN_NONSTD_SUFFIX_OFF
    H5T_CONV_Zf(FLOAT_COMPLEX, FLOAT16, H5_float_complex, H5__Float16, -FLT16_MAX, FLT16_MAX);
    H5_WARN_NONSTD_SUFFIX_ON
}
#endif

herr_t
H5T__conv_fcomplex_float(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zf(FLOAT_COMPLEX, FLOAT, H5_float_complex, float, -, -);
}

herr_t
H5T__conv_fcomplex_double(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zF(FLOAT_COMPLEX, DOUBLE, H5_float_complex, double, -, -);
}

herr_t
H5T__conv_fcomplex_ldouble(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zF(FLOAT_COMPLEX, LDOUBLE, H5_float_complex, long double, -, -);
}

herr_t
H5T__conv_fcomplex_dcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zZ(FLOAT_COMPLEX, DOUBLE_COMPLEX, H5_float_complex, H5_double_complex, -, -);
}

herr_t
H5T__conv_fcomplex_lcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zZ(FLOAT_COMPLEX, LDOUBLE_COMPLEX, H5_float_complex, H5_ldouble_complex, -, -);
}

herr_t
H5T__conv_dcomplex_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, SCHAR, H5_double_complex, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, UCHAR, H5_double_complex, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, SHORT, H5_double_complex, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, USHORT, H5_double_complex, unsigned short, 0, USHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, INT, H5_double_complex, int, INT_MIN, INT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, UINT, H5_double_complex, unsigned int, 0, UINT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, LONG, H5_double_complex, long, LONG_MIN, LONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, ULONG, H5_double_complex, unsigned long, 0, ULONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, LLONG, H5_double_complex, long long, LLONG_MIN, LLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_dcomplex_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(DOUBLE_COMPLEX, ULLONG, H5_double_complex, unsigned long long, 0, ULLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

#ifdef H5_HAVE__FLOAT16

herr_t
H5T__conv_dcomplex__Float16(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    
    H5_WARN_NONSTD_SUFFIX_OFF
    H5T_CONV_Zf(DOUBLE_COMPLEX, FLOAT16, H5_double_complex, H5__Float16, -FLT16_MAX, FLT16_MAX);
    H5_WARN_NONSTD_SUFFIX_ON
}
#endif

herr_t
H5T__conv_dcomplex_float(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Zf(DOUBLE_COMPLEX, FLOAT, H5_double_complex, float, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_dcomplex_double(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zf(DOUBLE_COMPLEX, DOUBLE, H5_double_complex, double, -, -);
}

herr_t
H5T__conv_dcomplex_ldouble(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zF(DOUBLE_COMPLEX, LDOUBLE, H5_double_complex, long double, -, -);
}

herr_t
H5T__conv_dcomplex_fcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Zz(DOUBLE_COMPLEX, FLOAT_COMPLEX, H5_double_complex, H5_float_complex, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_dcomplex_lcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zZ(DOUBLE_COMPLEX, LDOUBLE_COMPLEX, H5_double_complex, H5_ldouble_complex, -, -);
}

herr_t
H5T__conv_lcomplex_schar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, SCHAR, H5_ldouble_complex, signed char, SCHAR_MIN, SCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_uchar(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, UCHAR, H5_ldouble_complex, unsigned char, 0, UCHAR_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_short(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, SHORT, H5_ldouble_complex, short, SHRT_MIN, SHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_ushort(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, USHORT, H5_ldouble_complex, unsigned short, 0, USHRT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_int(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                       size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                       void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, INT, H5_ldouble_complex, int, INT_MIN, INT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_uint(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, UINT, H5_ldouble_complex, unsigned int, 0, UINT_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_long(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                        size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                        void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, LONG, H5_ldouble_complex, long, LONG_MIN, LONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

herr_t
H5T__conv_lcomplex_ulong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, ULONG, H5_ldouble_complex, unsigned long, 0, ULONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}

#ifdef H5T_CONV_INTERNAL_LDOUBLE_LLONG
herr_t
H5T__conv_lcomplex_llong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, LLONG, H5_ldouble_complex, long long, LLONG_MIN, LLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}
#endif 

#ifdef H5T_CONV_INTERNAL_LDOUBLE_ULLONG
herr_t
H5T__conv_lcomplex_ullong(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5_WARN_FLOAT_EQUAL_OFF
    H5T_CONV_Zx(LDOUBLE_COMPLEX, ULLONG, H5_ldouble_complex, unsigned long long, 0, ULLONG_MAX);
    H5_WARN_FLOAT_EQUAL_ON
}
#endif 

#ifdef H5_HAVE__FLOAT16
#ifdef H5T_CONV_INTERNAL_LDOUBLE_FLOAT16

herr_t
H5T__conv_lcomplex__Float16(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    
    H5_WARN_NONSTD_SUFFIX_OFF
    H5T_CONV_Zf(LDOUBLE_COMPLEX, FLOAT16, H5_ldouble_complex, H5__Float16, -FLT16_MAX, FLT16_MAX);
    H5_WARN_NONSTD_SUFFIX_ON
}
#endif
#endif

herr_t
H5T__conv_lcomplex_float(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata, const H5T_conv_ctx_t *conv_ctx,
                         size_t nelmts, size_t buf_stride, size_t H5_ATTR_UNUSED bkg_stride, void *buf,
                         void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Zf(LDOUBLE_COMPLEX, FLOAT, H5_ldouble_complex, float, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_lcomplex_double(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                          const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                          size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Zf(LDOUBLE_COMPLEX, DOUBLE, H5_ldouble_complex, double, -DBL_MAX, DBL_MAX);
}

herr_t
H5T__conv_lcomplex_ldouble(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                           const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                           size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_zf(LDOUBLE_COMPLEX, LDOUBLE, H5_ldouble_complex, long double, -, -);
}

herr_t
H5T__conv_lcomplex_fcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Zz(LDOUBLE_COMPLEX, FLOAT_COMPLEX, H5_ldouble_complex, H5_float_complex, -FLT_MAX, FLT_MAX);
}

herr_t
H5T__conv_lcomplex_dcomplex(const H5T_t *st, const H5T_t *dt, H5T_cdata_t *cdata,
                            const H5T_conv_ctx_t *conv_ctx, size_t nelmts, size_t buf_stride,
                            size_t H5_ATTR_UNUSED bkg_stride, void *buf, void H5_ATTR_UNUSED *bkg)
{
    H5T_CONV_Zz(LDOUBLE_COMPLEX, DOUBLE_COMPLEX, H5_ldouble_complex, H5_double_complex, -DBL_MAX, DBL_MAX);
}
#endif 
