/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5O_FRIEND      
#include "H5SMmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5Opkg.h"      
#include "H5SMpkg.h"     

typedef struct H5SM_compare_udata_t {
    const H5SM_mesg_key_t *key; 
    H5O_msg_crt_idx_t      idx; 
    herr_t                 ret; 
} H5SM_compare_udata_t;

static herr_t H5SM__compare_cb(const void *obj, size_t obj_len, void *udata);
static herr_t H5SM__compare_iter_op(H5O_t *oh, H5O_mesg_t *mesg, unsigned sequence, void *udata);

static herr_t
H5SM__compare_cb(const void *obj, size_t obj_len, void *_udata)
{
    H5SM_compare_udata_t *udata = (H5SM_compare_udata_t *)_udata;

    FUNC_ENTER_PACKAGE_NOERR

    
    if (udata->key->encoding_size > obj_len)
        udata->ret = 1;
    else if (udata->key->encoding_size < obj_len)
        udata->ret = -1;
    else
        
        udata->ret = memcmp(udata->key->encoding, obj, obj_len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5SM__compare_iter_op(H5O_t *oh, H5O_mesg_t *mesg , unsigned sequence, void *_udata )
{
    H5SM_compare_udata_t *udata     = (H5SM_compare_udata_t *)_udata;
    herr_t                ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(oh);
    assert(mesg);
    assert(udata && udata->key);

    
    if (sequence == udata->idx) {
        size_t aligned_encoded_size = H5O_ALIGN_OH(oh, udata->key->encoding_size);

        
        assert(mesg->raw_size > 0);

        if (aligned_encoded_size > mesg->raw_size)
            udata->ret = 1;
        else if (aligned_encoded_size < mesg->raw_size)
            udata->ret = -1;
        else {
            
            if (mesg->dirty)
                if (H5O_msg_flush(udata->key->file, oh, mesg) < 0)
                    HGOTO_ERROR(H5E_SOHM, H5E_CANTENCODE, H5_ITER_ERROR,
                                "unable to encode object header message");

            assert(udata->key->encoding_size <= mesg->raw_size);
            udata->ret = memcmp(udata->key->encoding, mesg->raw, udata->key->encoding_size);
        } 

        
        ret_value = H5_ITER_STOP;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5SM__message_compare(const void *rec1, const void *rec2, int *result)
{
    const H5SM_mesg_key_t *key       = (const H5SM_mesg_key_t *)rec1;
    const H5SM_sohm_t     *mesg      = (const H5SM_sohm_t *)rec2;
    herr_t                 ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (mesg->location == H5SM_IN_HEAP && key->message.location == H5SM_IN_HEAP) {
        if (key->message.u.heap_loc.fheap_id.val == mesg->u.heap_loc.fheap_id.val) {
            *result = 0;
            HGOTO_DONE(SUCCEED);
        }
    } 
    else if (mesg->location == H5SM_IN_OH && key->message.location == H5SM_IN_OH) {
        if (key->message.u.mesg_loc.oh_addr == mesg->u.mesg_loc.oh_addr &&
            key->message.u.mesg_loc.index == mesg->u.mesg_loc.index &&
            key->message.msg_type_id == mesg->msg_type_id) {
            *result = 0;
            HGOTO_DONE(SUCCEED);
        }
    } 

    
    if (key->message.hash > mesg->hash)
        *result = 1;
    else if (key->message.hash < mesg->hash)
        *result = -1;
    
    else {
        
        H5SM_compare_udata_t udata;

        assert(key->message.hash == mesg->hash);
        assert(key->encoding_size > 0 && key->encoding);

        
        udata.key = key;

        
        if (mesg->location == H5SM_IN_HEAP) {
            
            if (H5HF_op(key->fheap, &(mesg->u.heap_loc.fheap_id), H5SM__compare_cb, &udata) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPARE, FAIL, "can't compare btree2 records");
        } 
        else {
            H5O_loc_t           oloc; 
            H5O_mesg_operator_t op;   

            
            assert(key->file);
            assert(mesg->location == H5SM_IN_OH);

            
            if (H5O_loc_reset(&oloc) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "unable to initialize target location");

            
            oloc.file = key->file;
            oloc.addr = mesg->u.mesg_loc.oh_addr;

            
            udata.idx = mesg->u.mesg_loc.index;

            
            op.op_type  = H5O_MESG_OP_LIB;
            op.u.lib_op = H5SM__compare_iter_op;
            if (H5O_msg_iterate(&oloc, mesg->msg_type_id, &op, &udata) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "error iterating over links");
        } 

        *result = udata.ret;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5SM__message_encode(uint8_t *raw, const void *_nrecord, void *_ctx)
{
    H5SM_bt2_ctx_t    *ctx     = (H5SM_bt2_ctx_t *)_ctx; 
    const H5SM_sohm_t *message = (const H5SM_sohm_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    *raw++ = (uint8_t)message->location;
    UINT32ENCODE(raw, message->hash);

    if (message->location == H5SM_IN_HEAP) {
        UINT32ENCODE(raw, message->u.heap_loc.ref_count);
        H5MM_memcpy(raw, message->u.heap_loc.fheap_id.id, (size_t)H5O_FHEAP_ID_LEN);
    } 
    else {
        assert(message->location == H5SM_IN_OH);

        *raw++ = 0; 
        *raw++ = (uint8_t)message->msg_type_id;
        UINT16ENCODE(raw, message->u.mesg_loc.index);
        H5F_addr_encode_len((size_t)ctx->sizeof_addr, &raw, message->u.mesg_loc.oh_addr);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5SM__message_decode(const uint8_t *raw, void *_nrecord, void *_ctx)
{
    H5SM_bt2_ctx_t *ctx     = (H5SM_bt2_ctx_t *)_ctx; 
    H5SM_sohm_t    *message = (H5SM_sohm_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    message->location = (H5SM_storage_loc_t)*raw++;
    UINT32DECODE(raw, message->hash);

    if (message->location == H5SM_IN_HEAP) {
        UINT32DECODE(raw, message->u.heap_loc.ref_count);
        H5MM_memcpy(message->u.heap_loc.fheap_id.id, raw, (size_t)H5O_FHEAP_ID_LEN);
    } 
    else {
        assert(message->location == H5SM_IN_OH);

        raw++; 
        message->msg_type_id = *raw++;
        UINT16DECODE(raw, message->u.mesg_loc.index);
        H5F_addr_decode_len((size_t)ctx->sizeof_addr, &raw, &message->u.mesg_loc.oh_addr);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
