/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5FDprivate.h" 
#include "H5MMprivate.h" 
#include "H5Ppkg.h"      
#include "H5VMprivate.h" 

#define H5D_XFER_MAX_TEMP_BUF_SIZE sizeof(size_t)
#define H5D_XFER_MAX_TEMP_BUF_DEF  H5D_TEMP_BUF_SIZE
#define H5D_XFER_MAX_TEMP_BUF_ENC  H5P__encode_size_t
#define H5D_XFER_MAX_TEMP_BUF_DEC  H5P__decode_size_t

#define H5D_XFER_TCONV_BUF_SIZE sizeof(void *)
#define H5D_XFER_TCONV_BUF_DEF  NULL

#define H5D_XFER_BKGR_BUF_SIZE sizeof(void *)
#define H5D_XFER_BKGR_BUF_DEF  NULL

#define H5D_XFER_BKGR_BUF_TYPE_SIZE sizeof(H5T_bkg_t)
#define H5D_XFER_BKGR_BUF_TYPE_DEF  H5T_BKG_NO
#define H5D_XFER_BKGR_BUF_TYPE_ENC  H5P__dxfr_bkgr_buf_type_enc
#define H5D_XFER_BKGR_BUF_TYPE_DEC  H5P__dxfr_bkgr_buf_type_dec

#define H5D_XFER_BTREE_SPLIT_RATIO_SIZE sizeof(double[3])
#define H5D_XFER_BTREE_SPLIT_RATIO_DEF                                                                       \
    {                                                                                                        \
        0.1, 0.5, 0.9                                                                                        \
    }
#define H5D_XFER_BTREE_SPLIT_RATIO_ENC H5P__dxfr_btree_split_ratio_enc
#define H5D_XFER_BTREE_SPLIT_RATIO_DEC H5P__dxfr_btree_split_ratio_dec

#define H5D_XFER_VLEN_ALLOC_SIZE sizeof(H5MM_allocate_t)
#define H5D_XFER_VLEN_ALLOC_DEF  H5D_VLEN_ALLOC

#define H5D_XFER_VLEN_ALLOC_INFO_SIZE sizeof(void *)
#define H5D_XFER_VLEN_ALLOC_INFO_DEF  H5D_VLEN_ALLOC_INFO

#define H5D_XFER_VLEN_FREE_SIZE sizeof(H5MM_free_t)
#define H5D_XFER_VLEN_FREE_DEF  H5D_VLEN_FREE

#define H5D_XFER_VLEN_FREE_INFO_SIZE sizeof(void *)
#define H5D_XFER_VLEN_FREE_INFO_DEF  H5D_VLEN_FREE_INFO

#define H5D_XFER_HYPER_VECTOR_SIZE_SIZE sizeof(size_t)
#define H5D_XFER_HYPER_VECTOR_SIZE_DEF  H5D_IO_VECTOR_SIZE
#define H5D_XFER_HYPER_VECTOR_SIZE_ENC  H5P__encode_size_t
#define H5D_XFER_HYPER_VECTOR_SIZE_DEC  H5P__decode_size_t

#define H5D_XFER_IO_XFER_MODE_SIZE sizeof(H5FD_mpio_xfer_t)
#define H5D_XFER_IO_XFER_MODE_DEF  H5FD_MPIO_INDEPENDENT
#define H5D_XFER_IO_XFER_MODE_ENC  H5P__dxfr_io_xfer_mode_enc
#define H5D_XFER_IO_XFER_MODE_DEC  H5P__dxfr_io_xfer_mode_dec

#define H5D_XFER_MPIO_COLLECTIVE_OPT_SIZE  sizeof(H5FD_mpio_collective_opt_t)
#define H5D_XFER_MPIO_COLLECTIVE_OPT_DEF   H5FD_MPIO_COLLECTIVE_IO
#define H5D_XFER_MPIO_COLLECTIVE_OPT_ENC   H5P__dxfr_mpio_collective_opt_enc
#define H5D_XFER_MPIO_COLLECTIVE_OPT_DEC   H5P__dxfr_mpio_collective_opt_dec
#define H5D_XFER_MPIO_CHUNK_OPT_HARD_SIZE  sizeof(H5FD_mpio_chunk_opt_t)
#define H5D_XFER_MPIO_CHUNK_OPT_HARD_DEF   H5FD_MPIO_CHUNK_DEFAULT
#define H5D_XFER_MPIO_CHUNK_OPT_HARD_ENC   H5P__dxfr_mpio_chunk_opt_hard_enc
#define H5D_XFER_MPIO_CHUNK_OPT_HARD_DEC   H5P__dxfr_mpio_chunk_opt_hard_dec
#define H5D_XFER_MPIO_CHUNK_OPT_NUM_SIZE   sizeof(unsigned)
#define H5D_XFER_MPIO_CHUNK_OPT_NUM_DEF    H5D_ONE_LINK_CHUNK_IO_THRESHOLD
#define H5D_XFER_MPIO_CHUNK_OPT_NUM_ENC    H5P__encode_unsigned
#define H5D_XFER_MPIO_CHUNK_OPT_NUM_DEC    H5P__decode_unsigned
#define H5D_XFER_MPIO_CHUNK_OPT_RATIO_SIZE sizeof(unsigned)
#define H5D_XFER_MPIO_CHUNK_OPT_RATIO_DEF  H5D_MULTI_CHUNK_IO_COL_THRESHOLD
#define H5D_XFER_MPIO_CHUNK_OPT_RATIO_ENC  H5P__encode_unsigned
#define H5D_XFER_MPIO_CHUNK_OPT_RATIO_DEC  H5P__decode_unsigned

#define H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_SIZE sizeof(H5D_mpio_actual_chunk_opt_mode_t)
#define H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_DEF  H5D_MPIO_NO_CHUNK_OPTIMIZATION

#define H5D_MPIO_ACTUAL_IO_MODE_SIZE sizeof(H5D_mpio_actual_io_mode_t)
#define H5D_MPIO_ACTUAL_IO_MODE_DEF  H5D_MPIO_NO_COLLECTIVE

#define H5D_MPIO_NO_COLLECTIVE_CAUSE_SIZE sizeof(uint32_t)
#define H5D_MPIO_NO_COLLECTIVE_CAUSE_DEF  H5D_MPIO_COLLECTIVE

#define H5D_XFER_EDC_SIZE sizeof(H5Z_EDC_t)
#define H5D_XFER_EDC_DEF  H5Z_ENABLE_EDC
#define H5D_XFER_EDC_ENC  H5P__dxfr_edc_enc
#define H5D_XFER_EDC_DEC  H5P__dxfr_edc_dec

#define H5D_XFER_FILTER_CB_SIZE sizeof(H5Z_cb_t)
#define H5D_XFER_FILTER_CB_DEF                                                                               \
    {                                                                                                        \
        NULL, NULL                                                                                           \
    }

#define H5D_XFER_CONV_CB_SIZE sizeof(H5T_conv_cb_t)
#define H5D_XFER_CONV_CB_DEF                                                                                 \
    {                                                                                                        \
        NULL, NULL                                                                                           \
    }

#define H5D_XFER_XFORM_SIZE  sizeof(void *)
#define H5D_XFER_XFORM_DEF   NULL
#define H5D_XFER_XFORM_SET   H5P__dxfr_xform_set
#define H5D_XFER_XFORM_GET   H5P__dxfr_xform_get
#define H5D_XFER_XFORM_ENC   H5P__dxfr_xform_enc
#define H5D_XFER_XFORM_DEC   H5P__dxfr_xform_dec
#define H5D_XFER_XFORM_DEL   H5P__dxfr_xform_del
#define H5D_XFER_XFORM_COPY  H5P__dxfr_xform_copy
#define H5D_XFER_XFORM_CMP   H5P__dxfr_xform_cmp
#define H5D_XFER_XFORM_CLOSE H5P__dxfr_xform_close

#define H5D_XFER_DSET_IO_SEL_SIZE  sizeof(H5S_t *)
#define H5D_XFER_DSET_IO_SEL_DEF   NULL
#define H5D_XFER_DSET_IO_SEL_COPY  H5P__dxfr_dset_io_hyp_sel_copy
#define H5D_XFER_DSET_IO_SEL_CMP   H5P__dxfr_dset_io_hyp_sel_cmp
#define H5D_XFER_DSET_IO_SEL_CLOSE H5P__dxfr_dset_io_hyp_sel_close
#ifdef QAK
#define H5D_XFER_DSET_IO_SEL_ENC H5P__dxfr_edc_enc
#define H5D_XFER_DSET_IO_SEL_DEC H5P__dxfr_edc_dec
#endif 

#define H5D_XFER_SELECTION_IO_MODE_SIZE sizeof(H5D_selection_io_mode_t)
#define H5D_XFER_SELECTION_IO_MODE_DEF  H5D_SELECTION_IO_MODE_DEFAULT
#define H5D_XFER_SELECTION_IO_MODE_ENC  H5P__dxfr_selection_io_mode_enc
#define H5D_XFER_SELECTION_IO_MODE_DEC  H5P__dxfr_selection_io_mode_dec

#define H5D_XFER_NO_SELECTION_IO_CAUSE_SIZE sizeof(uint32_t)
#define H5D_XFER_NO_SELECTION_IO_CAUSE_DEF  0

#define H5D_XFER_ACTUAL_SELECTION_IO_MODE_SIZE sizeof(uint32_t)
#define H5D_XFER_ACTUAL_SELECTION_IO_MODE_DEF  0

#define H5D_XFER_MODIFY_WRITE_BUF_SIZE sizeof(bool)
#define H5D_XFER_MODIFY_WRITE_BUF_DEF  false
#define H5D_XFER_MODIFY_WRITE_BUF_ENC  H5P__dxfr_modify_write_buf_enc
#define H5D_XFER_MODIFY_WRITE_BUF_DEC  H5P__dxfr_modify_write_buf_dec

static herr_t H5P__dxfr_reg_prop(H5P_genclass_t *pclass);

static herr_t H5P__dxfr_bkgr_buf_type_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_bkgr_buf_type_dec(const void **pp, void *value);
static herr_t H5P__dxfr_btree_split_ratio_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_btree_split_ratio_dec(const void **pp, void *value);
static herr_t H5P__dxfr_io_xfer_mode_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_io_xfer_mode_dec(const void **pp, void *value);
static herr_t H5P__dxfr_mpio_collective_opt_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_mpio_collective_opt_dec(const void **pp, void *value);
static herr_t H5P__dxfr_mpio_chunk_opt_hard_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_mpio_chunk_opt_hard_dec(const void **pp, void *value);
static herr_t H5P__dxfr_edc_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_edc_dec(const void **pp, void *value);
static herr_t H5P__dxfr_xform_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dxfr_xform_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dxfr_xform_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_xform_dec(const void **pp, void *value);
static herr_t H5P__dxfr_xform_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dxfr_xform_copy(const char *name, size_t size, void *value);
static int    H5P__dxfr_xform_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__dxfr_xform_close(const char *name, size_t size, void *value);
static herr_t H5P__dxfr_dset_io_hyp_sel_copy(const char *name, size_t size, void *value);
static int    H5P__dxfr_dset_io_hyp_sel_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__dxfr_dset_io_hyp_sel_close(const char *name, size_t size, void *value);
static herr_t H5P__dxfr_selection_io_mode_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_selection_io_mode_dec(const void **pp, void *value);
static herr_t H5P__dxfr_modify_write_buf_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dxfr_modify_write_buf_dec(const void **pp, void *value);

const H5P_libclass_t H5P_CLS_DXFR[1] = {{
    "data transfer",       
    H5P_TYPE_DATASET_XFER, 

    &H5P_CLS_ROOT_g,            
    &H5P_CLS_DATASET_XFER_g,    
    &H5P_CLS_DATASET_XFER_ID_g, 
    &H5P_LST_DATASET_XFER_ID_g, 
    H5P__dxfr_reg_prop,         

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static const size_t H5D_def_max_temp_buf_g =
    H5D_XFER_MAX_TEMP_BUF_DEF; 
static const void *H5D_def_tconv_buf_g =
    H5D_XFER_TCONV_BUF_DEF; 
static const void     *H5D_def_bkgr_buf_g = H5D_XFER_BKGR_BUF_DEF; 
static const H5T_bkg_t H5D_def_bkgr_buf_type_g = H5D_XFER_BKGR_BUF_TYPE_DEF;
static const double    H5D_def_btree_split_ratio_g[3] =
    H5D_XFER_BTREE_SPLIT_RATIO_DEF; 
static const H5MM_allocate_t H5D_def_vlen_alloc_g =
    H5D_XFER_VLEN_ALLOC_DEF; 
static const void *H5D_def_vlen_alloc_info_g =
    H5D_XFER_VLEN_ALLOC_INFO_DEF; 
static const H5MM_free_t H5D_def_vlen_free_g =
    H5D_XFER_VLEN_FREE_DEF; 
static const void *H5D_def_vlen_free_info_g =
    H5D_XFER_VLEN_FREE_INFO_DEF; 
static const size_t H5D_def_hyp_vec_size_g =
    H5D_XFER_HYPER_VECTOR_SIZE_DEF; 
static const H5FD_mpio_xfer_t H5D_def_io_xfer_mode_g =
    H5D_XFER_IO_XFER_MODE_DEF; 
static const H5FD_mpio_chunk_opt_t      H5D_def_mpio_chunk_opt_mode_g      = H5D_XFER_MPIO_CHUNK_OPT_HARD_DEF;
static const H5FD_mpio_collective_opt_t H5D_def_mpio_collective_opt_mode_g = H5D_XFER_MPIO_COLLECTIVE_OPT_DEF;
static const unsigned                   H5D_def_mpio_chunk_opt_num_g       = H5D_XFER_MPIO_CHUNK_OPT_NUM_DEF;
static const unsigned                   H5D_def_mpio_chunk_opt_ratio_g = H5D_XFER_MPIO_CHUNK_OPT_RATIO_DEF;
static const H5D_mpio_actual_chunk_opt_mode_t H5D_def_mpio_actual_chunk_opt_mode_g =
    H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_DEF;
static const H5D_mpio_actual_io_mode_t      H5D_def_mpio_actual_io_mode_g = H5D_MPIO_ACTUAL_IO_MODE_DEF;
static const H5D_mpio_no_collective_cause_t H5D_def_mpio_no_collective_cause_g =
    H5D_MPIO_NO_COLLECTIVE_CAUSE_DEF;
static const H5Z_EDC_t H5D_def_enable_edc_g = H5D_XFER_EDC_DEF;       
static const H5Z_cb_t  H5D_def_filter_cb_g  = H5D_XFER_FILTER_CB_DEF; 
static const H5T_conv_cb_t H5D_def_conv_cb_g =
    H5D_XFER_CONV_CB_DEF; 
static const void  *H5D_def_xfer_xform_g = H5D_XFER_XFORM_DEF; 
static const H5S_t *H5D_def_dset_io_sel_g =
    H5D_XFER_DSET_IO_SEL_DEF; 
static const H5D_selection_io_mode_t H5D_def_selection_io_mode_g     = H5D_XFER_SELECTION_IO_MODE_DEF;
static const uint32_t                H5D_def_no_selection_io_cause_g = H5D_XFER_NO_SELECTION_IO_CAUSE_DEF;
static const uint32_t H5D_def_actual_selection_io_mode_g             = H5D_XFER_ACTUAL_SELECTION_IO_MODE_DEF;
static const bool     H5D_def_modify_write_buf_g                     = H5D_XFER_MODIFY_WRITE_BUF_DEF;

static herr_t
H5P__dxfr_reg_prop(H5P_genclass_t *pclass)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5D_XFER_MAX_TEMP_BUF_NAME, H5D_XFER_MAX_TEMP_BUF_SIZE,
                           &H5D_def_max_temp_buf_g, NULL, NULL, NULL, H5D_XFER_MAX_TEMP_BUF_ENC,
                           H5D_XFER_MAX_TEMP_BUF_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_TCONV_BUF_NAME, H5D_XFER_TCONV_BUF_SIZE, &H5D_def_tconv_buf_g,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_BKGR_BUF_NAME, H5D_XFER_BKGR_BUF_SIZE, &H5D_def_bkgr_buf_g, NULL,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_BKGR_BUF_TYPE_NAME, H5D_XFER_BKGR_BUF_TYPE_SIZE,
                           &H5D_def_bkgr_buf_type_g, NULL, NULL, NULL, H5D_XFER_BKGR_BUF_TYPE_ENC,
                           H5D_XFER_BKGR_BUF_TYPE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_BTREE_SPLIT_RATIO_NAME, H5D_XFER_BTREE_SPLIT_RATIO_SIZE,
                           H5D_def_btree_split_ratio_g, NULL, NULL, NULL, H5D_XFER_BTREE_SPLIT_RATIO_ENC,
                           H5D_XFER_BTREE_SPLIT_RATIO_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_VLEN_ALLOC_NAME, H5D_XFER_VLEN_ALLOC_SIZE, &H5D_def_vlen_alloc_g,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_VLEN_ALLOC_INFO_NAME, H5D_XFER_VLEN_ALLOC_INFO_SIZE,
                           &H5D_def_vlen_alloc_info_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                           NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_VLEN_FREE_NAME, H5D_XFER_VLEN_FREE_SIZE, &H5D_def_vlen_free_g,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_VLEN_FREE_INFO_NAME, H5D_XFER_VLEN_FREE_INFO_SIZE,
                           &H5D_def_vlen_free_info_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                           NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_HYPER_VECTOR_SIZE_NAME, H5D_XFER_HYPER_VECTOR_SIZE_SIZE,
                           &H5D_def_hyp_vec_size_g, NULL, NULL, NULL, H5D_XFER_HYPER_VECTOR_SIZE_ENC,
                           H5D_XFER_HYPER_VECTOR_SIZE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_IO_XFER_MODE_NAME, H5D_XFER_IO_XFER_MODE_SIZE,
                           &H5D_def_io_xfer_mode_g, NULL, NULL, NULL, H5D_XFER_IO_XFER_MODE_ENC,
                           H5D_XFER_IO_XFER_MODE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");
    if (H5P__register_real(pclass, H5D_XFER_MPIO_COLLECTIVE_OPT_NAME, H5D_XFER_MPIO_COLLECTIVE_OPT_SIZE,
                           &H5D_def_mpio_collective_opt_mode_g, NULL, NULL, NULL,
                           H5D_XFER_MPIO_COLLECTIVE_OPT_ENC, H5D_XFER_MPIO_COLLECTIVE_OPT_DEC, NULL, NULL,
                           NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");
    if (H5P__register_real(pclass, H5D_XFER_MPIO_CHUNK_OPT_HARD_NAME, H5D_XFER_MPIO_CHUNK_OPT_HARD_SIZE,
                           &H5D_def_mpio_chunk_opt_mode_g, NULL, NULL, NULL, H5D_XFER_MPIO_CHUNK_OPT_HARD_ENC,
                           H5D_XFER_MPIO_CHUNK_OPT_HARD_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");
    if (H5P__register_real(pclass, H5D_XFER_MPIO_CHUNK_OPT_NUM_NAME, H5D_XFER_MPIO_CHUNK_OPT_NUM_SIZE,
                           &H5D_def_mpio_chunk_opt_num_g, NULL, NULL, NULL, H5D_XFER_MPIO_CHUNK_OPT_NUM_ENC,
                           H5D_XFER_MPIO_CHUNK_OPT_NUM_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");
    if (H5P__register_real(pclass, H5D_XFER_MPIO_CHUNK_OPT_RATIO_NAME, H5D_XFER_MPIO_CHUNK_OPT_RATIO_SIZE,
                           &H5D_def_mpio_chunk_opt_ratio_g, NULL, NULL, NULL,
                           H5D_XFER_MPIO_CHUNK_OPT_RATIO_ENC, H5D_XFER_MPIO_CHUNK_OPT_RATIO_DEC, NULL, NULL,
                           NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_NAME, H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_SIZE,
                           &H5D_def_mpio_actual_chunk_opt_mode_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                           NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_MPIO_ACTUAL_IO_MODE_NAME, H5D_MPIO_ACTUAL_IO_MODE_SIZE,
                           &H5D_def_mpio_actual_io_mode_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                           NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_MPIO_LOCAL_NO_COLLECTIVE_CAUSE_NAME, H5D_MPIO_NO_COLLECTIVE_CAUSE_SIZE,
                           &H5D_def_mpio_no_collective_cause_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                           NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_MPIO_GLOBAL_NO_COLLECTIVE_CAUSE_NAME,
                           H5D_MPIO_NO_COLLECTIVE_CAUSE_SIZE, &H5D_def_mpio_no_collective_cause_g, NULL, NULL,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_EDC_NAME, H5D_XFER_EDC_SIZE, &H5D_def_enable_edc_g, NULL, NULL,
                           NULL, H5D_XFER_EDC_ENC, H5D_XFER_EDC_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_FILTER_CB_NAME, H5D_XFER_FILTER_CB_SIZE, &H5D_def_filter_cb_g,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_CONV_CB_NAME, H5D_XFER_CONV_CB_SIZE, &H5D_def_conv_cb_g, NULL,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_XFORM_NAME, H5D_XFER_XFORM_SIZE, &H5D_def_xfer_xform_g, NULL,
                           H5D_XFER_XFORM_SET, H5D_XFER_XFORM_GET, H5D_XFER_XFORM_ENC, H5D_XFER_XFORM_DEC,
                           H5D_XFER_XFORM_DEL, H5D_XFER_XFORM_COPY, H5D_XFER_XFORM_CMP,
                           H5D_XFER_XFORM_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_DSET_IO_SEL_NAME, H5D_XFER_DSET_IO_SEL_SIZE,
                           &H5D_def_dset_io_sel_g, NULL, NULL, NULL, NULL, NULL, NULL,
                           H5D_XFER_DSET_IO_SEL_COPY, H5D_XFER_DSET_IO_SEL_CMP,
                           H5D_XFER_DSET_IO_SEL_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    if (H5P__register_real(pclass, H5D_XFER_SELECTION_IO_MODE_NAME, H5D_XFER_SELECTION_IO_MODE_SIZE,
                           &H5D_def_selection_io_mode_g, NULL, NULL, NULL, H5D_XFER_SELECTION_IO_MODE_ENC,
                           H5D_XFER_SELECTION_IO_MODE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_NO_SELECTION_IO_CAUSE_NAME, H5D_XFER_NO_SELECTION_IO_CAUSE_SIZE,
                           &H5D_def_no_selection_io_cause_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
                           NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_XFER_ACTUAL_SELECTION_IO_MODE_NAME,
                           H5D_XFER_ACTUAL_SELECTION_IO_MODE_SIZE, &H5D_def_actual_selection_io_mode_g, NULL,
                           NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_XFER_MODIFY_WRITE_BUF_NAME, H5D_XFER_MODIFY_WRITE_BUF_SIZE,
                           &H5D_def_modify_write_buf_g, NULL, NULL, NULL, H5D_XFER_MODIFY_WRITE_BUF_ENC,
                           H5D_XFER_MODIFY_WRITE_BUF_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_bkgr_buf_type_enc(const void *value, void **_pp, size_t *size)
{
    const H5T_bkg_t *bkgr_buf_type = (const H5T_bkg_t *)value; 
    uint8_t        **pp            = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(bkgr_buf_type);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*bkgr_buf_type;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_bkgr_buf_type_dec(const void **_pp, void *_value)
{
    H5T_bkg_t      *bkgr_buf_type = (H5T_bkg_t *)_value; 
    const uint8_t **pp            = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(bkgr_buf_type);

    
    *bkgr_buf_type = (H5T_bkg_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_btree_split_ratio_enc(const void *value, void **_pp, size_t *size)
{
    const double *btree_split_ratio = (const double *)value; 
    uint8_t     **pp                = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(btree_split_ratio);
    assert(size);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)sizeof(double);

        
        H5_ENCODE_DOUBLE(*pp, *(const double *)btree_split_ratio);
        btree_split_ratio++;

        
        H5_ENCODE_DOUBLE(*pp, *(const double *)btree_split_ratio);
        btree_split_ratio++;

        
        H5_ENCODE_DOUBLE(*pp, *(const double *)btree_split_ratio);
    } 

    
    *size += 1 + (3 * sizeof(double));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_btree_split_ratio_dec(const void **_pp, void *_value)
{
    double         *btree_split_ratio = (double *)_value; 
    unsigned        enc_size;                             
    const uint8_t **pp        = (const uint8_t **)_pp;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(btree_split_ratio);

    
    enc_size = *(*pp)++;
    if (enc_size != sizeof(double))
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "double value can't be decoded");

    
    H5_DECODE_DOUBLE(*pp, btree_split_ratio[0]);
    H5_DECODE_DOUBLE(*pp, btree_split_ratio[1]);
    H5_DECODE_DOUBLE(*pp, btree_split_ratio[2]);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size,
                    void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5Z_xform_copy((H5Z_data_xform_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "error copying the data transform info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size,
                    void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5Z_xform_copy((H5Z_data_xform_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "error copying the data transform info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_enc(const void *value, void **_pp, size_t *size)
{
    const H5Z_data_xform_t *data_xform_prop =
        *(const H5Z_data_xform_t *const *)value; 
    const char *pexp      = NULL;                
    size_t      len       = 0;                   
    uint8_t   **pp        = (uint8_t **)_pp;
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(size);

    
    if (NULL != data_xform_prop) {
        
        if (NULL == (pexp = H5Z_xform_extract_xform_str(data_xform_prop)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "failed to retrieve transform expression");

        
        len = strlen(pexp) + 1;
    } 

    if (NULL != *pp) {
        uint64_t enc_value;
        unsigned enc_size;

        
        enc_value = (uint64_t)len;
        enc_size  = H5VM_limit_enc_size(enc_value);
        assert(enc_size < 256);
        *(*pp)++ = (uint8_t)enc_size;
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);

        if (NULL != data_xform_prop) {
            
            assert(pexp);

            
            H5MM_memcpy(*pp, (const uint8_t *)pexp, len);
            *pp += len;
            *pp[0] = '\0';
        } 
    }     

    
    *size += (1 + H5VM_limit_enc_size((uint64_t)len));
    if (NULL != pexp)
        *size += len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_dec(const void **_pp, void *_value)
{
    H5Z_data_xform_t **data_xform_prop = (H5Z_data_xform_t **)_value; 
    size_t             len;                                           
    const uint8_t    **pp = (const uint8_t **)_pp;
    unsigned           enc_size;
    uint64_t           enc_value;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pp);
    assert(*pp);
    assert(data_xform_prop);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    len = (size_t)enc_value;

    if (0 != len) {
        if (NULL == (*data_xform_prop = H5Z_xform_create((const char *)*pp)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "unable to create data transform info");
        *pp += len;
    } 
    else
        *data_xform_prop = H5D_XFER_XFORM_DEF;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size,
                    void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(value);

    if (H5Z_xform_destroy(*(H5Z_data_xform_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL, "error closing the parse tree");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(value);

    
    if (H5Z_xform_copy((H5Z_data_xform_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "error copying the data transform info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__dxfr_xform_cmp(const void *_xform1, const void *_xform2, size_t H5_ATTR_UNUSED size)
{
    const H5Z_data_xform_t *const *xform1 =
        (const H5Z_data_xform_t *const *)_xform1; 
    const H5Z_data_xform_t *const *xform2 =
        (const H5Z_data_xform_t *const *)_xform2; 
    const char *pexp1, *pexp2;                    
    herr_t      ret_value = 0;                    

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(xform1);
    assert(xform2);
    assert(size == sizeof(H5Z_data_xform_t *));

    
    if (*xform1 == NULL && *xform2 != NULL)
        HGOTO_DONE(-1);
    if (*xform1 != NULL && *xform2 == NULL)
        HGOTO_DONE(1);

    if (*xform1) {
        assert(*xform2);

        
        pexp1 = H5Z_xform_extract_xform_str(*xform1);
        pexp2 = H5Z_xform_extract_xform_str(*xform2);

        
        if (pexp1 == NULL && pexp2 != NULL)
            HGOTO_DONE(-1);
        if (pexp1 != NULL && pexp2 == NULL)
            HGOTO_DONE(1);

        if (pexp1) {
            assert(pexp2);
            ret_value = strcmp(pexp1, pexp2);
        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_xform_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(value);

    if (H5Z_xform_destroy(*(H5Z_data_xform_t **)value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL, "error closing the parse tree");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pset_data_transform(hid_t plist_id, const char *expression)
{
    H5P_genplist_t   *plist;                     
    H5Z_data_xform_t *data_xform_prop = NULL;    
    herr_t            ret_value       = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (expression == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "expression cannot be NULL");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_XFER_XFORM_NAME, &data_xform_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "error getting data transform expression");

    
    if (H5Z_xform_destroy(data_xform_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to release data transform expression");

    
    if (NULL == (data_xform_prop = H5Z_xform_create(expression)))
        HGOTO_ERROR(H5E_PLIST, H5E_NOSPACE, FAIL, "unable to create data transform info");

    
    if (H5P_poke(plist, H5D_XFER_XFORM_NAME, &data_xform_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "Error setting data transform expression");

done:
    if (ret_value < 0)
        if (data_xform_prop && H5Z_xform_destroy(data_xform_prop) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to release data transform expression");

    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Pget_data_transform(hid_t plist_id, char *expression , size_t size)
{
    H5P_genplist_t   *plist;                  
    H5Z_data_xform_t *data_xform_prop = NULL; 
    size_t            len;
    const char       *pexp;
    ssize_t           ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (H5P_peek(plist, H5D_XFER_XFORM_NAME, &data_xform_prop) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "error getting data transform expression");

    if (NULL == data_xform_prop)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "data transform has not been set");

    
    if (NULL == (pexp = H5Z_xform_extract_xform_str(data_xform_prop)))
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "failed to retrieve transform expression");

    
    len = strlen(pexp);
    if (expression) {
        strncpy(expression, pexp, size);
        if (len >= size)
            expression[size - 1] = '\0';
    } 

    ret_value = (ssize_t)len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_buffer(hid_t plist_id, size_t size, void *tconv, void *bkg)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (size == 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "buffer size must not be zero");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_XFER_MAX_TEMP_BUF_NAME, &size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "Can't set transfer buffer size");
    if (H5P_set(plist, H5D_XFER_TCONV_BUF_NAME, &tconv) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "Can't set transfer type conversion buffer");
    if (H5P_set(plist, H5D_XFER_BKGR_BUF_NAME, &bkg) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "Can't set background type conversion buffer");

done:
    FUNC_LEAVE_API(ret_value)
} 

size_t
H5Pget_buffer(hid_t plist_id, void **tconv , void **bkg )
{
    H5P_genplist_t *plist;     
    size_t          size;      
    size_t          ret_value; 

    FUNC_ENTER_API(0)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, 0, "can't find object for ID");

    
    if (tconv)
        if (H5P_get(plist, H5D_XFER_TCONV_BUF_NAME, tconv) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, 0, "Can't get transfer type conversion buffer");
    if (bkg)
        if (H5P_get(plist, H5D_XFER_BKGR_BUF_NAME, bkg) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, 0, "Can't get background type conversion buffer");

    
    if (H5P_get(plist, H5D_XFER_MAX_TEMP_BUF_NAME, &size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, 0, "Can't set transfer buffer size");

    
    ret_value = size;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_preserve(hid_t plist_id, bool status)
{
    H5P_genplist_t *plist;               
    H5T_bkg_t       need_bkg;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    need_bkg = status ? H5T_BKG_YES : H5T_BKG_NO;
    if (H5P_set(plist, H5D_XFER_BKGR_BUF_TYPE_NAME, &need_bkg) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5Pget_preserve(hid_t plist_id)
{
    H5T_bkg_t       need_bkg;  
    H5P_genplist_t *plist;     
    int             ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5D_XFER_BKGR_BUF_TYPE_NAME, &need_bkg) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

    
    ret_value = need_bkg ? true : false;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_edc_check(hid_t plist_id, H5Z_EDC_t check)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (check != H5Z_ENABLE_EDC && check != H5Z_DISABLE_EDC)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a valid value");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_XFER_EDC_NAME, &check) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

H5Z_EDC_t
H5Pget_edc_check(hid_t plist_id)
{
    H5P_genplist_t *plist;     
    H5Z_EDC_t       ret_value; 

    FUNC_ENTER_API(H5Z_ERROR_EDC)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, H5Z_ERROR_EDC, "can't find object for ID");

    
    if (H5P_get(plist, H5D_XFER_EDC_NAME, &ret_value) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, H5Z_ERROR_EDC, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_filter_callback(hid_t plist_id, H5Z_filter_func_t func, void *op_data)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 
    H5Z_cb_t        cb_struct;

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    cb_struct.func    = func;
    cb_struct.op_data = op_data;

    if (H5P_set(plist, H5D_XFER_FILTER_CB_NAME, &cb_struct) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Pset_type_conv_cb(hid_t plist_id, H5T_conv_except_func_t op, void *operate_data)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 
    H5T_conv_cb_t   cb_struct;

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    cb_struct.func      = op;
    cb_struct.user_data = operate_data;

    if (H5P_set(plist, H5D_XFER_CONV_CB_NAME, &cb_struct) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Pget_type_conv_cb(hid_t plist_id, H5T_conv_except_func_t *op , void **operate_data )
{
    H5P_genplist_t *plist; 
    H5T_conv_cb_t   cb_struct;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5D_XFER_CONV_CB_NAME, &cb_struct) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

    
    *op           = cb_struct.func;
    *operate_data = cb_struct.user_data;

done:
    FUNC_LEAVE_API(ret_value)
}

herr_t
H5Pget_btree_ratios(hid_t plist_id, double *left , double *middle , double *right )
{
    H5P_genplist_t *plist;                
    double          btree_split_ratio[3]; 
    herr_t          ret_value = SUCCEED;  

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5D_XFER_BTREE_SPLIT_RATIO_NAME, &btree_split_ratio) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

    
    if (left)
        *left = btree_split_ratio[0];
    if (middle)
        *middle = btree_split_ratio[1];
    if (right)
        *right = btree_split_ratio[2];

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_btree_ratios(hid_t plist_id, double left, double middle, double right)
{
    H5P_genplist_t *plist;               
    double          split_ratio[3];      
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (left < 0.0 || left > 1.0 || middle < 0.0 || middle > 1.0 || right < 0.0 || right > 1.0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "split ratio must satisfy 0.0 <= X <= 1.0");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    split_ratio[0] = left;
    split_ratio[1] = middle;
    split_ratio[2] = right;

    
    if (H5P_set(plist, H5D_XFER_BTREE_SPLIT_RATIO_NAME, &split_ratio) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5P_set_vlen_mem_manager(H5P_genplist_t *plist, H5MM_allocate_t alloc_func, void *alloc_info,
                         H5MM_free_t free_func, void *free_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    assert(plist);

    
    if (H5P_set(plist, H5D_XFER_VLEN_ALLOC_NAME, &alloc_func) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");
    if (H5P_set(plist, H5D_XFER_VLEN_ALLOC_INFO_NAME, &alloc_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");
    if (H5P_set(plist, H5D_XFER_VLEN_FREE_NAME, &free_func) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");
    if (H5P_set(plist, H5D_XFER_VLEN_FREE_INFO_NAME, &free_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Pset_vlen_mem_manager(hid_t plist_id, H5MM_allocate_t alloc_func, void *alloc_info, H5MM_free_t free_func,
                        void *free_info)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a dataset transfer property list");

    
    if (H5P_set_vlen_mem_manager(plist, alloc_func, alloc_info, free_func, free_info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set values");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_vlen_mem_manager(hid_t plist_id, H5MM_allocate_t *alloc_func , void **alloc_info ,
                        H5MM_free_t *free_func , void **free_info )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (alloc_func)
        if (H5P_get(plist, H5D_XFER_VLEN_ALLOC_NAME, alloc_func) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");
    if (alloc_info)
        if (H5P_get(plist, H5D_XFER_VLEN_ALLOC_INFO_NAME, alloc_info) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");
    if (free_func)
        if (H5P_get(plist, H5D_XFER_VLEN_FREE_NAME, free_func) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");
    if (free_info)
        if (H5P_get(plist, H5D_XFER_VLEN_FREE_INFO_NAME, free_info) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_hyper_vector_size(hid_t plist_id, size_t vector_size)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (vector_size < 1)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "vector size too small");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_XFER_HYPER_VECTOR_SIZE_NAME, &vector_size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_hyper_vector_size(hid_t plist_id, size_t *vector_size )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (vector_size)
        if (H5P_get(plist, H5D_XFER_HYPER_VECTOR_SIZE_NAME, vector_size) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__dxfr_io_xfer_mode_enc(const void *value, void **_pp, size_t *size)
{
    const H5FD_mpio_xfer_t *xfer_mode = (const H5FD_mpio_xfer_t *)value; 
    uint8_t               **pp        = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(xfer_mode);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*xfer_mode;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_io_xfer_mode_dec(const void **_pp, void *_value)
{
    H5FD_mpio_xfer_t *xfer_mode = (H5FD_mpio_xfer_t *)_value; 
    const uint8_t   **pp        = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(xfer_mode);

    
    *xfer_mode = (H5FD_mpio_xfer_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_mpio_collective_opt_enc(const void *value, void **_pp, size_t *size)
{
    const H5FD_mpio_collective_opt_t *coll_opt =
        (const H5FD_mpio_collective_opt_t *)value; 
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(coll_opt);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*coll_opt;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_mpio_collective_opt_dec(const void **_pp, void *_value)
{
    H5FD_mpio_collective_opt_t *coll_opt =
        (H5FD_mpio_collective_opt_t *)_value; 
    const uint8_t **pp = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(coll_opt);

    
    *coll_opt = (H5FD_mpio_collective_opt_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_mpio_chunk_opt_hard_enc(const void *value, void **_pp, size_t *size)
{
    const H5FD_mpio_chunk_opt_t *chunk_opt =
        (const H5FD_mpio_chunk_opt_t *)value; 
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(chunk_opt);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*chunk_opt;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_mpio_chunk_opt_hard_dec(const void **_pp, void *_value)
{
    H5FD_mpio_chunk_opt_t *chunk_opt = (H5FD_mpio_chunk_opt_t *)_value; 
    const uint8_t        **pp        = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(chunk_opt);

    
    *chunk_opt = (H5FD_mpio_chunk_opt_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

#ifdef H5_HAVE_PARALLEL

herr_t
H5Pget_mpio_actual_chunk_opt_mode(hid_t                             plist_id,
                                  H5D_mpio_actual_chunk_opt_mode_t *actual_chunk_opt_mode )
{
    H5P_genplist_t *plist;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (actual_chunk_opt_mode)
        if (H5P_get(plist, H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_NAME, actual_chunk_opt_mode) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_mpio_actual_io_mode(hid_t plist_id, H5D_mpio_actual_io_mode_t *actual_io_mode )
{
    H5P_genplist_t *plist;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (actual_io_mode)
        if (H5P_get(plist, H5D_MPIO_ACTUAL_IO_MODE_NAME, actual_io_mode) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_mpio_no_collective_cause(hid_t plist_id, uint32_t *local_no_collective_cause ,
                                uint32_t *global_no_collective_cause )
{
    H5P_genplist_t *plist;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (local_no_collective_cause)
        if (H5P_get(plist, H5D_MPIO_LOCAL_NO_COLLECTIVE_CAUSE_NAME, local_no_collective_cause) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get local value");
    if (global_no_collective_cause)
        if (H5P_get(plist, H5D_MPIO_GLOBAL_NO_COLLECTIVE_CAUSE_NAME, global_no_collective_cause) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get global value");

done:
    FUNC_LEAVE_API(ret_value)
} 
#endif 

static herr_t
H5P__dxfr_edc_enc(const void *value, void **_pp, size_t *size)
{
    const H5Z_EDC_t *check = (const H5Z_EDC_t *)value; 
    uint8_t        **pp    = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(check);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*check;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_edc_dec(const void **_pp, void *_value)
{
    H5Z_EDC_t      *check = (H5Z_EDC_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(check);

    
    *check = (H5Z_EDC_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_dset_io_hyp_sel_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    H5S_t *orig_space = *(H5S_t **)value; 
    H5S_t *new_space  = NULL;             
    herr_t ret_value  = SUCCEED;          

    FUNC_ENTER_PACKAGE

    
    if (orig_space) {
        
        if (NULL == (new_space = H5S_copy(orig_space, false, true)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "error copying the dataset I/O selection");

        
        *(void **)value = new_space;
    } 

done:
    
    if (ret_value < 0)
        if (new_space && H5S_close(new_space) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL, "error closing dataset I/O selection dataspace");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5P__dxfr_dset_io_hyp_sel_cmp(const void *_space1, const void *_space2, size_t H5_ATTR_UNUSED size)
{
    const H5S_t *const *space1    = (const H5S_t *const *)_space1; 
    const H5S_t *const *space2    = (const H5S_t *const *)_space2; 
    herr_t              ret_value = 0;                             

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(space1);
    assert(space1);
    assert(size == sizeof(H5S_t *));

    
    if (*space1 == NULL && *space2 != NULL)
        HGOTO_DONE(-1);
    if (*space1 != NULL && *space2 == NULL)
        HGOTO_DONE(1);

    if (*space1) {
        assert(*space2);

        
        
        if (true != H5S_extent_equal(*space1, *space2))
            HGOTO_DONE(-1);

        
        H5_WARN_CAST_AWAY_CONST_OFF
        if (true != H5S_select_shape_same((H5S_t *)*space1, (H5S_t *)*space2))
            HGOTO_DONE(-1);
        H5_WARN_CAST_AWAY_CONST_ON
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_dset_io_hyp_sel_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *_value)
{
    H5S_t *space     = *(H5S_t **)_value; 
    herr_t ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    
    if (space && H5S_close(space) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTCLOSEOBJ, FAIL, "error closing dataset I/O selection dataspace");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dxfr_selection_io_mode_enc(const void *value, void **_pp, size_t *size)
{
    const H5D_selection_io_mode_t *select_io_mode =
        (const H5D_selection_io_mode_t *)value; 
    uint8_t **pp = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(select_io_mode);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*select_io_mode;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_selection_io_mode_dec(const void **_pp, void *_value)
{
    H5D_selection_io_mode_t *select_io_mode = (H5D_selection_io_mode_t *)_value; 
    const uint8_t          **pp             = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(select_io_mode);

    
    *select_io_mode = (H5D_selection_io_mode_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_dataset_io_hyperslab_selection(hid_t plist_id, unsigned rank, H5S_seloper_t op, const hsize_t start[],
                                      const hsize_t stride[], const hsize_t count[], const hsize_t block[])
{
    H5P_genplist_t *plist               = NULL;    
    H5S_t          *space               = NULL;    
    bool            space_created       = false;   
    bool            reset_prop_on_error = false;   
    herr_t          ret_value           = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (rank < 1 || rank > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid rank value: %u", rank);
    if (!(op > H5S_SELECT_NOOP && op < H5S_SELECT_INVALID))
        HGOTO_ERROR(H5E_ARGS, H5E_UNSUPPORTED, FAIL, "invalid selection operation");
    if (start == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "'count' pointer is NULL");
    if (stride != NULL) {
        unsigned u; 

        
        for (u = 0; u < rank; u++)
            if (stride[u] == 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid value - stride[%u]==0", u);
    } 
    if (count == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "'start' pointer is NULL");
    

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_XFER_DSET_IO_SEL_NAME, &space) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "error getting dataset I/O selection");

    
    if (NULL != space) {
        int sndims; 

        
        if ((sndims = H5S_GET_EXTENT_NDIMS(space)) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get selection's dataspace rank");

        
        if ((unsigned)sndims != rank) {
            
            if (op == H5S_SELECT_SET) {
                
                if (H5S_close(space) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to release dataspace");

                
                space = NULL;

                
                reset_prop_on_error = true;
            } 
            else
                HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "different rank for previous and new selections");
        } 
    }     

    
    if (NULL == space) {
        hsize_t  dims[H5S_MAX_RANK]; 
        unsigned u;                  

        
        for (u = 0; u < rank; u++)
            dims[u] = (H5S_UNLIMITED - 1);

        
        if (NULL == (space = H5S_create_simple(rank, dims, NULL)))
            HGOTO_ERROR(H5E_PLIST, H5E_CANTCREATE, FAIL, "unable to create dataspace for selection");
        space_created = true;
    } 

    
    if (H5S_select_hyperslab(space, op, start, stride, count, block) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSELECT, FAIL, "can't create selection");

    
    if (H5P_poke(plist, H5D_XFER_DSET_IO_SEL_NAME, &space) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "error setting dataset I/O selection");
    space_created = false; 

done:
    
    if (ret_value < 0) {
        if (reset_prop_on_error && plist && H5P_poke(plist, H5D_XFER_DSET_IO_SEL_NAME, &space) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "error setting dataset I/O selection");
        if (space_created && H5S_close(space) < 0)
            HDONE_ERROR(H5E_PLIST, H5E_CLOSEERROR, FAIL, "unable to release dataspace");
    } 

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_selection_io(hid_t plist_id, H5D_selection_io_mode_t selection_io_mode)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (plist_id == H5P_DEFAULT)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't set values in default property list");

    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a dxpl");

    
    if (H5P_set(plist, H5D_XFER_SELECTION_IO_MODE_NAME, &selection_io_mode) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_selection_io(hid_t plist_id, H5D_selection_io_mode_t *selection_io_mode )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a dxpl");

    
    if (selection_io_mode)
        if (H5P_get(plist, H5D_XFER_SELECTION_IO_MODE_NAME, selection_io_mode) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_no_selection_io_cause(hid_t plist_id, uint32_t *no_selection_io_cause )
{
    H5P_genplist_t *plist;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (no_selection_io_cause)
        if (H5P_get(plist, H5D_XFER_NO_SELECTION_IO_CAUSE_NAME, no_selection_io_cause) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get no_selection_io_cause value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_actual_selection_io_mode(hid_t plist_id, uint32_t *actual_selection_io_mode )
{
    H5P_genplist_t *plist;
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (actual_selection_io_mode)
        if (H5P_get(plist, H5D_XFER_ACTUAL_SELECTION_IO_MODE_NAME, actual_selection_io_mode) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get actual_selection_io_mode value");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__dxfr_modify_write_buf_enc(const void *value, void **_pp , size_t *size )
{
    const bool *modify_write_buf = (const bool *)value; 
    uint8_t   **pp               = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(modify_write_buf);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)(!!(*modify_write_buf));

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dxfr_modify_write_buf_dec(const void **_pp, void *_value )
{
    bool           *modify_write_buf = (bool *)_value; 
    const uint8_t **pp               = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(modify_write_buf);

    
    *modify_write_buf = (bool)*(*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_modify_write_buf(hid_t plist_id, bool modify_write_buf)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (plist_id == H5P_DEFAULT)
        HGOTO_ERROR(H5E_PLIST, H5E_BADVALUE, FAIL, "can't set values in default property list");

    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, false)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a dxpl");

    
    if (H5P_set(plist, H5D_XFER_MODIFY_WRITE_BUF_NAME, &modify_write_buf) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_modify_write_buf(hid_t plist_id, bool *modify_write_buf )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_XFER, true)))
        HGOTO_ERROR(H5E_PLIST, H5E_BADTYPE, FAIL, "not a dxpl");

    
    if (modify_write_buf)
        if (H5P_get(plist, H5D_XFER_MODIFY_WRITE_BUF_NAME, modify_write_buf) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 
