% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cdc.R
\name{ext_bmiz}
\alias{ext_bmiz}
\title{Calculate extended BMI measures}
\usage{
ext_bmiz(
  data,
  age = "agem",
  wt = "wt",
  ht = "ht",
  bmi = "bmi",
  adjust.integer.age = TRUE,
  ref.data.path = ""
)
}
\arguments{
\item{data}{Input data frame or data table}

\item{age}{Name of input column containing subject age in months in quotes, default "agem"}

\item{wt}{Name of input column containing weight (kg) value in quotes, default "wt"}

\item{ht}{Name of input column containing height (cm) value in quotes, default "ht"}

\item{bmi}{Name of input column containing calculated BMI in quotes, default "bmi"}

\item{adjust.integer.age}{If age inputs are all integer, add 0.5 if TRUE;
default TRUE}

\item{ref.data.path}{Path to directory containing reference data}
}
\value{
Expanded data frame containing computed BMI values
}
\description{
\code{ext_bmiz} Calculates the sigma (scale parameter for the half-normal
distribution), extended BMI percentile, extended BMIz, and the CDC LMS
Z-scores for weight, height, and BMI for children between 2 and 19.9 years
of age. Note that for BMIs <= 95th percentile of the CDC growth charts, the
extended values for BMI are equal to the LMS values. The extended values
differ only for children who have a BMI > 95th percentile.
}
\details{
This function should produce output equivalent to the SAS macro provided at
https://www.cdc.gov/nccdphp/dnpao/growthcharts/resources/sas.htm. The macro
was updated in December, 2022, according to the findings of the NCHS
report available at https://dx.doi.org/10.15620/cdc:121711. This function has
been updated to match it as of growthcleanr v2.1.0.

The extended BMIz is the inverse cumulative distribution function (CDF) of
the extended BMI percentile. If the extended percentile is very close to
100, the \code{qnorm} function in R produces an infinite value. This occurs
only if the the extended BMI percentile is > 99.99999999999999. This occurs
infrequently, such as a 48-month-old with a BMI > 39, and it is likely that
these BMIs represent data entry errors. For these cases, extended BMIz is
set to 8.21, a value that is slightly greater than the largest value that
can be calculated.

See the \code{README.md} file for descriptions of the output columns
generated by this function.

\code{data} must have columns for at least age, sex, weight, height, and bmi.

\code{age} should be coded in months, using the most precise values available.
To convert to months from age in years, multiply by 12. To convert to months
from age in days, divide by 30.4375 (365.25 / 12).

\code{sex} is coded as 1, boys, Boys, b, B, males, Males, m, or M for male
subjects or 2, girls, Girls, g, G, females, Females, f, or F for female
subjects. Note that this is different from \code{cleangrowth}, which uses
0 (Male) and 1 (Female).

\code{wt} should be in kilograms.

\code{ht} should be in centimeters.

Specify the input data parameter names for \code{age}, \code{wt},
\code{ht}, \code{bmi} using quotation marks. See example below.

If the parameter \code{adjust.integer.age} is \code{TRUE} (the default),
0.5 will be added to all \code{age} if all input values are integers. Set to
\code{FALSE} to disable.

By default, the reference data file \code{CDCref_d.csv}, made available at
https://www.cdc.gov/nccdphp/dnpao/growthcharts/resources/sas.htm, is included
in this package for convenience. If you are developing this package, use
\code{ref.data.path} to adjust the path to this file from your working
directory if necessary.
}
\examples{
\donttest{
# Run on a small subset of given data
df <- as.data.frame(syngrowth)
df <- df[df$subjid \%in\% unique(df[, "subjid"])[1:5], ]
df <- cbind(df,
            "gcr_result" = cleangrowth(df$subjid,
                                       df$param,
                                       df$agedays,
                                       df$sex,
                                       df$measurement))
df_wide <- longwide(df) # convert to wide format for ext_bmiz
df_wide_bmi <- simple_bmi(df_wide) # compute simple BMI

# Calling the function with default column names
df_bmiz <- ext_bmiz(df_wide_bmi)

# Specifying different column names; note that quotes are used
dfc <- simple_bmi(df_wide)
colnames(dfc)[colnames(dfc) \%in\% c("agem", "wt", "ht")] <-
  c("agemos", "weightkg", "heightcm")
df_bmiz <- ext_bmiz(dfc, age="agemos", wt="weightkg", ht="heightcm")

# Disabling conversion of all-integer age in months to (age + 0.5)
dfc <- simple_bmi(df_wide)
df_bmiz <- ext_bmiz(dfc, adjust.integer.age=FALSE)
}
}
