#' Read Light Data from .lux Files
#'
#' Reads and parses light intensity data from .lux files generated by
#' geolocation loggers. Handles various formats and automatically detects
#' the start of data.
#'
#' @param file_path Character string specifying the path to the .lux file
#'
#' @return A data.frame with columns:
#'   \item{Date}{POSIXct datetime in UTC}
#'   \item{Light}{Numeric light intensity in lux}
#'
#' @examples
#' # Read example data included with package
#' example_file <- gls_example("W086")
#' light_data <- read_lux_file(example_file)
#' head(light_data)
#'
#' @export
#' @importFrom stringr str_match
read_lux_file <- function(file_path) {

  if (!file.exists(file_path)) {
    stop("File does not exist: ", file_path)
  }

  lines <- readLines(file_path, warn = FALSE)

  # Find data start (line with "DD/MM/YYYY HH:MM:SS" and "light")
  data_start <- which(grepl("DD/MM/YYYY HH:MM:SS.*light", lines, ignore.case = TRUE))[1] + 1

  if (is.na(data_start)) {
    stop("Could not find data header in file")
  }

  data_lines <- lines[data_start:length(lines)]
  data_lines <- data_lines[nchar(data_lines) > 0]

  # Parse date and light values
  parsed <- str_match(data_lines,
                      "^(\\d{2}/\\d{2}/\\d{4}\\s+\\d{2}:\\d{2}:\\d{2})\\s+([0-9.]+)")

  df <- data.frame(
    Date = as.POSIXct(parsed[, 2], format = "%d/%m/%Y %H:%M:%S", tz = "UTC"),
    Light = as.numeric(parsed[, 3])
  )

  # Remove NA rows
  df <- df[!is.na(df$Date) & !is.na(df$Light), ]

  if (nrow(df) == 0) {
    stop("No valid data found in file")
  }

  return(df)
}
