% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\name{residuals.gnm}
\alias{residuals.gnm}
\title{Residuals for Generalized Nonlinear Models}
\usage{
\method{residuals}{gnm}(
  object,
  type = c("quantile", "deviance", "pearson"),
  standardized = FALSE,
  plot.it = FALSE,
  identify,
  dispersion = NULL,
  ...
)
}
\arguments{
\item{object}{a object of the class \emph{gnm}.}

\item{type}{an (optional) character string giving the type of residuals which should be returned. The available options are: (1) "quantile", (2) "deviance", and (3) "pearson". As default, \code{type} is set to "quantile".}

\item{standardized}{an (optional) logical switch indicating if the residuals should be standardized by dividing by the square root of \eqn{(1-h)}, where \eqn{h} is a measure of leverage. As default, \code{standardized} is set to \code{FALSE}.}

\item{plot.it}{an (optional) logical switch indicating if a plot of the residuals versus the fitted values is required. As default, \code{plot.it} is set to \code{FALSE}.}

\item{identify}{an (optional) integer value indicating the number of individuals to identify on the plot of residuals. This is only appropriate when \code{plot.it=TRUE}.}

\item{dispersion}{an (optional) value indicating the dispersion parameter estimate that must be used to calculate residuals.}

\item{...}{further arguments passed to or from other methods}
}
\value{
A vector with the observed residuals type \code{type}.
}
\description{
Computes residuals for a fitted generalized nonlinear model.
}
\examples{
###### Example 1: The effects of fertilizers on coastal Bermuda grass
data(Grass)
fit1 <- gnm(Yield ~ b0 + b1/(Nitrogen + a1) + b2/(Phosphorus + a2) + b3/(Potassium + a3),
            family=gaussian(inverse), start=c(b0=0.1,b1=13,b2=1,b3=1,a1=45,a2=15,a3=30), data=Grass)
residuals(fit1, type="quantile", plot.it=TRUE, col="red", pch=20, col.lab="blue",
                col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 2: Assay of an Insecticide with a Synergist
data(Melanopus)
fit2 <- gnm(Killed/Exposed ~ b0 + b1*log(Insecticide-a1) + b2*Synergist/(a2 + Synergist),
            family=binomial(logit), weights=Exposed, start=c(b0=-3,b1=1.2,a1=1.7,b2=1.7,a2=2),
		   data=Melanopus)
residuals(fit2, type="quantile", plot.it=TRUE, col="red", pch=20, col.lab="blue",
                col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 3: Developmental rate of Drosophila melanogaster
data(Drosophila)
fit3 <- gnm(Duration ~ b0 + b1*Temp + b2/(Temp-a), family=Gamma(log),
            start=c(b0=3,b1=-0.25,b2=-210,a=55), weights=Size, data=Drosophila)
residuals(fit3, type="quantile", plot.it=TRUE, col="red", pch=20, col.lab="blue",
                col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 4: Radioimmunological Assay of Cortisol
data(Cortisol)
fit4 <- gnm(Y ~ b0 + (b1-b0)/(1 + exp(b2+ b3*lDose))^b4, family=Gamma(identity),
            start=c(b0=130,b1=2800,b2=3,b3=3,b4=0.5), data=Cortisol)
residuals(fit4, type="quantile", plot.it=TRUE, col="red", pch=20, col.lab="blue",
                col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 5: Age and Eye Lens Weight of Rabbits in Australia
data(rabbits)
fit5 <- gnm(wlens ~ b1 - b2/(age + b3), family=Gamma(log),
            start=c(b1=5.5,b2=130,b3=35), data=rabbits)
residuals(fit5, type="quantile", plot.it=TRUE, col="red", pch=20, col.lab="blue",
                col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 6: Calls to a technical support help line
data(calls)
fit6 <- gnm(calls ~ SSlogis(week, Asym, xmid, scal), family=poisson(identity), data=calls)
residuals(fit6, type="quantile", plot.it=TRUE, col="red", pch=20, col.lab="blue",
                col.axis="blue", col.main="black", family="mono", cex=0.8)

}
\references{
Atkinson A.C. (1985) \emph{Plots, Transformations and Regression}. Oxford University Press, Oxford.

Davison A.C., Gigli A. (1989) Deviance Residuals and Normal Scores Plots. \emph{Biometrika} 76, 211-221.

Dunn P.K., Smyth G.K. (1996) Randomized Quantile Residuals. \emph{Journal of Computational and Graphical Statistics} 5, 236-244.

Pierce D.A., Schafer D.W. (1986) Residuals in Generalized Linear Models. \emph{Journal of the American Statistical Association} 81, 977-986.
}
