#' Visual Parameters for gg1d Plots
#'
#' Configures aesthetic and layout settings for plots generated by `gg1d`.
#'
#' @param show_legend Display the legend on the plot (flag).
#' @param show_legend_titles Display titles for legends (flag).
#' @param legend_position Position of the legend ("right", "left", "bottom", "top").
#' @param legend_title_position Position of the legend title ("top", "bottom", "left", "right").
#' @param legend_nrow Number of rows in the legend (number).
#' @param legend_ncol Number of columns in the legend. If set, `legend_nrow` should be `NULL` (number).
#' @param legend_title_size Size of the legend title text (number).
#' @param legend_text_size Size of the text within the legend (number).
#' @param legend_key_size Size of the legend key symbols (number).
#' @param beautify_text Beautify y-axis text and legend titles by capitalizing words and adding spaces (flag).
#' @param vertical_spacing Space between each data row in points (number).
#' @param fontsize_barplot_y_numbers fontsize of the text describing numeric barplot max & min values (number).
#' @param max_digits_barplot_y_numbers Number of digits to round the numeric barplot max and min values to (number).
#' @param fontsize_y_title fontsize of the y axis titles (a.k.a the data.frame column names) (number).
#' @param y_axis_position Position of the y-axis ("left" or "right").
#' @param numeric_plot_type Type of visualization for numeric data: "bar" or "heatmap".
#' @param colours_default Default colors for categorical variables without a custom palette.
#' @param colours_default_logical Colors for binary variables: a vector of three colors representing `TRUE`, `FALSE`, and `NA` respectively (character).
#' @param colours_missing Color for missing (`NA`) values in categorical plots (string).
#' @param na_marker Text used to mark `NA` values in numeric plots (string).
#' @param na_marker_size Size of the text marker for `NA` values (number).
#' @param na_marker_colour Color of the `NA` text marker (string).
#' @param colours_heatmap_low Color for the lowest value in heatmaps (string).
#' @param colours_heatmap_high Color for the highest value in heatmaps (string).
#' @param transform_heatmap Transformation to apply before visualizing heatmap values ("identity", "log10", "log2").
#' @param show_na_marker_categorical Show a marker for `NA` values on categorical tiles (flag).
#' @param show_na_marker_heatmap Show a marker for `NA` values on heatmap tiles (flag).
#' @param show_values_heatmap Display numerical values on heatmap tiles (flag).
#' @param fontsize_values_heatmap Font size for heatmap values (number).
#' @param legend_orientation_heatmap should legend orientation be "horizontal" or "vertical".
#' @param width controls how much space is present between bars and tiles within each plot. Can be 0-1 where values of 1 makes bars/tiles take up 100% of available space (no gaps between bars).
#' @param relative_height_numeric how many times taller should numeric plots be relative to categorical tile plots. Only taken into account if numeric_plot_type == "bar" (number)
#' @param cli_header Text used for h1 header. Included so it can be tweaked by packages that use gg1d, so they can customise how the info messages appear.
#' @param interactive_svg_width,interactive_svg_height width and height of the interactive graphic region (in inches). Only used when `interactive = TRUE`.
#' @param colours_values_heatmap Color for heatmap values (string).
#' @return A list of visualization parameters for `gg1d`.
#' @export
#' @inherit gg1d examples
gg1d_options <- function(
    # Default Colours
    colours_default = c("#66C2A5", "#FC8D62", "#8DA0CB", "#E78AC3", "#A6D854", "#FFD92F", "#E5C494"),
    colours_default_logical = c("TRUE" = "#648fff", "FALSE" = "#dc267f"),
    colours_missing = "grey90",
    # Legend
    show_legend_titles = FALSE,
    legend_title_position = c("top", "bottom", "left", "right"),
    legend_nrow = 4, legend_ncol = NULL,
    legend_title_size = NULL, legend_text_size = NULL, legend_key_size = 0.3,
    legend_orientation_heatmap = c("horizontal", "vertical"),
    show_legend = TRUE,
    legend_position = c("right", "left", "bottom", "top"),
    # Missing Data
    na_marker = "!", na_marker_size = 8, na_marker_colour = "black",
    show_na_marker_categorical = FALSE,
    show_na_marker_heatmap = FALSE,
    # Heatmap
    colours_heatmap_low = "purple",
    colours_heatmap_high = "seagreen",
    transform_heatmap = c("identity", "log10", "log2"),
    fontsize_values_heatmap = 3,
    show_values_heatmap = FALSE,
    colours_values_heatmap = "white",
    # Global Paramaters
    vertical_spacing = 0,
    numeric_plot_type = c("bar", "heatmap"),
    y_axis_position = c("left", "right"),
    width = 0.9,
    relative_height_numeric = 4,
    cli_header = "Running gg1d",
    # Interactivity
    interactive_svg_width = NULL,
    interactive_svg_height = NULL,
    # Text
    fontsize_barplot_y_numbers = 8,
    max_digits_barplot_y_numbers = 3,
    fontsize_y_title = 12,
    beautify_text = TRUE) {
  # Legend-related
  assertions::assert_flag(show_legend)
  assertions::assert_flag(show_legend_titles)
  assertions::assert_number(legend_key_size)
  if (!is.null(legend_nrow)) assertions::assert_number(legend_nrow)
  if (!is.null(legend_ncol)) assertions::assert_number(legend_ncol)
  if (!is.null(legend_title_size)) assertions::assert_number(legend_title_size)
  if (!is.null(legend_ncol) & !is.null(legend_nrow)) {
    cli::cli_warn(c("!" = "Both {.arg legend_ncol} and {.arg legend_nrow} were supplied. {.arg legend_nrow} will be ignored.  Explicitly set one of these arguments to NULL to avoid this warning message"))
    legend_nrow <- NULL
  }

  # Colors and styling
  assertions::assert_string(colours_missing)
  assertions::assert_string(colours_heatmap_low)
  assertions::assert_string(colours_heatmap_high)
  assertions::assert_string(colours_values_heatmap)
  assertions::assert_equal(length(colours_default_logical), 2)
  assertions::assert_names_include(colours_default_logical, c("TRUE", "FALSE"))

  # NA markers and visualization
  assertions::assert_logical(show_na_marker_categorical)
  assertions::assert_logical(show_na_marker_heatmap)
  assertions::assert_logical(show_values_heatmap)
  assertions::assert_string(na_marker_colour)

  # Text and layout
  assertions::assert_flag(beautify_text)
  assertions::assert_number(vertical_spacing)
  assertions::assert_number(fontsize_values_heatmap)
  assertions::assert_number(relative_height_numeric)

  # Argument matching (these check the allowed values for these parameters)
  legend_position <- rlang::arg_match(legend_position)
  legend_title_position <- rlang::arg_match(legend_title_position)
  y_axis_position <- rlang::arg_match(y_axis_position)
  numeric_plot_type <- rlang::arg_match(numeric_plot_type)
  legend_orientation_heatmap <- rlang::arg_match(legend_orientation_heatmap)
  transform_heatmap <- rlang::arg_match(transform_heatmap)

  # Interactive
  if (!is.null(interactive_svg_width)) {
    assertions::assert_number(interactive_svg_width)
  }
  if (!is.null(interactive_svg_height)) {
    assertions::assert_number(interactive_svg_height)
  }

  # Ignore relative_height_numeric if plot type is numeric
  if (numeric_plot_type == "heatmap") relative_height_numeric <- 1

  # Create options list
  opts <- list(
    show_legend = show_legend,
    show_legend_titles = show_legend_titles,
    legend_position = legend_position,
    legend_title_position = legend_title_position,
    legend_nrow = legend_nrow,
    legend_ncol = legend_ncol,
    legend_title_size = legend_title_size,
    legend_text_size = legend_text_size,
    legend_key_size = legend_key_size,
    legend_orientation_heatmap = legend_orientation_heatmap,
    beautify_text = beautify_text,
    vertical_spacing = vertical_spacing,
    fontsize_barplot_y_numbers = fontsize_barplot_y_numbers,
    max_digits_barplot_y_numbers = max_digits_barplot_y_numbers,
    fontsize_y_title = fontsize_y_title,
    y_axis_position = y_axis_position,
    numeric_plot_type = numeric_plot_type,
    colours_default = colours_default,
    colours_default_logical = colours_default_logical,
    colours_missing = colours_missing,
    cli_header = cli_header,
    na_marker = na_marker,
    na_marker_size = na_marker_size,
    na_marker_colour = na_marker_colour,
    colours_heatmap_low = colours_heatmap_low,
    colours_heatmap_high = colours_heatmap_high,
    transform_heatmap = transform_heatmap,
    show_na_marker_categorical = show_na_marker_categorical,
    show_na_marker_heatmap = show_na_marker_heatmap,
    show_values_heatmap = show_values_heatmap,
    fontsize_values_heatmap = fontsize_values_heatmap,
    colours_values_heatmap = colours_values_heatmap,
    relative_height_numeric = relative_height_numeric,
    width = width,
    interactive_svg_width = interactive_svg_width,
    interactive_svg_height = interactive_svg_height
  )

  # Set class so we can check what is supplied to gg1d options argument was produced
  # specifically by this function
  class(opts) <- "gg1d_options"

  return(opts)
}
