#' Plot Fitted Probability Distribution for S-FFA
#'
#' Generates a plot showing the probability density of a stationary model (left panel) 
#' and the data (right panel).
#'
#' @param results A fitted flood frequency model generated by [fit_lmoments()],
#' [fit_mle()] or [fit_gmle()].
#'
#' @param show_line If `TRUE` (default), draw a fitted line through the data.
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return `ggplot`; a plot showing:
#' - The likelihood function of the distribution plotted vertically on the left panel.
#' - The data, connected with a line if `show_line == TRUE`, on the right panel.
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' years <- seq(from = 1901, to = 2000)
#' results <- fit_lmoments(data, "WEI")
#' plot_sffa_fit(results)
#'
#' @import ggplot2
#' @export
plot_sffa_fit <- function(
	results,
	show_line = TRUE,
	...
) {

	# Define y-axis range
	y_range <- c(0, 1.1 * max(results$data))

	# Create a dataframe for the distribution
	quantiles <- seq(0, 1.1 * max(results$data), length.out = 200)

	density <- sapply(quantiles, function(x) {
		exp(utils_log_likelihood(x, results$distribution, results$params))
	})

	distribution_df <- data.frame(x = quantiles, y = density)

	# Create a dataframe for the data
	data_df <- data.frame(x = seq_along(results$data), y = results$data)

	# Capture optional arguments
	args <- list(...)

	# Set labels and title 
	title <- args$title %||% "Fitted Flood Frequency Model (S-FFA)"
	xlabel <- args$xlabel %||% "Index"
	ylabel <- args$ylabel %||% expression(Streamflow ~ m^3/s)

	# Plot of distribution
	p1 <- ggplot(distribution_df, aes(x = .data$x, y = .data$y)) +
		geom_line(color = "steelblue", linewidth = 1) +
		geom_area(fill = "steelblue", alpha = 0.5) +
		labs(x = ylabel, y = "Density") +
		coord_flip()

	p1 <- add_theme(p1) + theme(
		axis.text.x = element_blank(),
		axis.ticks.x = element_blank(),
	)

	# Plot of data
	p2 <- ggplot(data_df, aes(x = .data$x, y = .data$y)) +
		geom_point(color = "dimgray", size = 2.25) + 
		{ if (show_line) geom_line(color = "dimgray", linewidth = 1) else NULL } +
		scale_y_continuous(limits = y_range) +
		labs(x = xlabel, y = "", title = title)

	p2 <- add_theme(p2) + theme(
		axis.text.y = element_blank(),
		axis.ticks.y = element_blank()
	)

	# Return the plot
	(p1 + p2) + 
	plot_layout(widths = c(1, 6)) + 
	plot_annotation(title = "")

} 

