#' genDateTime
#
#' @description genDateTime generates a date-time object from the single components date and time.
#
#' @param refOrEsDf a data.frame. Either the reference dataset or the event sampling raw dataset (already merged to a single dataset).
#
#' @param refOrEs a character string. Enter "REF" if the argument refOrEs is the reference dataset, enter "ES" if it is the event sampling dataset.
#
#' @param dateFormat a character string. Choose the current date format, "ymd" (default), "mdy", or "dmy".
#
#' @param timeFormat  a character string. Choose the current time format, "HMS" (default), or "HM".
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @param RELEVANTINFO_ES a list. This list is generated by function \code{\link{setES}}.
#
#' @param RELEVANTVN_REF a list. This list is generated by function \code{\link{setREF}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @details Depending on whether the ESM dataset(s) or the reference dataset are passed as the first argument the start date and start time (same for end date and end time) doesn't have the same meaning. See definition of the date and time relating arguments in \code{\link{setES}} and \code{\link{setREF}}.
#
#' @return The dataset that was passed as first argument with two additional columns START_DATETIME and END_DATETIME, i.e. the date-time objects of the separated date and time of both ESM start and ESM end. See \strong{Details} for more information.
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute genDateTime. Start --------------
#' relEs <- relevantESVN(svyName="survey_name", IMEI="IMEI",
#' STARTDATE="start_date", STARTTIME="start_time",
#' ENDDATE="end_date", ENDTIME="end_time")
#' imeiNumbers <- as.character(referenceDf$imei)
#' surveyNames <- c("morningTestGroup", "dayTestGroup", "eveningTestGroup",
#' "morningControlGroup", "dayControlGroup", "eveningControlGroup")
#' RELEVANT_ES <- setES(4, imeiNumbers, surveyNames, relEs)
#' RELEVANTINFO_ES <- RELEVANT_ES[["RELEVANTINFO_ES"]]
#' RELEVANTVN_ES <- RELEVANT_ES[["RELEVANTVN_ES"]]
#' esLs <- esList(list(morningControl, dayControl, eveningControl,
#' morningTest, dayTest, eveningTest), RELEVANTVN_ES)
#' keyLs <- genKey(esLs)
#' relRef <- relevantREFVN(ID="id", IMEI="imei", ST="st",
#' STARTDATE="start_date", STARTTIME="start_time",
#' ENDDATE="end_date", ENDTIME="end_time")
#' RELEVANTVN_REF <- setREF(4, relRef)
#' # Prerequisites in order to execute genDateTime. End ----------------
#' # ------------------------------------------------------
#' # Run function 7 of 29; see esmprep functions' hierarchy.
#' # ------------------------------------------------------
#' # Applying function to reference dataset (7a of 29)
#' referenceDfList <- genDateTime(referenceDf, "REF", RELEVANTINFO_ES, RELEVANTVN_ES,
#' RELEVANTVN_REF)
#' 
#' # Extract reference dataset from output
#' referenceDfNew <- referenceDfList[["refOrEsDf"]]
#' 
#' # Extract extended list of relevant variables names of reference dataset
#' RELEVANTVN_REF <- referenceDfList[["extendedVNList"]]
#' 
#' # Applying function to raw ESM dataset(s) (7b of 29)
#' # keyLs is the result of function 'genKey'.
#' keyList <- genDateTime(keyLs, "ES", RELEVANTINFO_ES, RELEVANTVN_ES,
#' RELEVANTVN_REF)
#' 
#' # Extract list of raw ESM datasets from output
#' keyLsNew <- keyList[["refOrEsDf"]]
#' 
#' # Extract extended list of relevant variables names of raw ESM datasets
#' RELEVANTVN_ES <- keyList[["extendedVNList"]]
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 7 of 29).\cr \code{genDateTime} is the reverse function of \code{\link{splitDateTime}}.
#
#' @export
#
genDateTime <- function(refOrEsDf=NULL, refOrEs = NULL, RELEVANTINFO_ES = NULL, RELEVANTVN_ES = NULL, RELEVANTVN_REF = NULL, dateFormat = "ymd", timeFormat = "HMS") {

    # ERROR handling start -------------------------------------------------
    if ( (refOrEs != "REF" & refOrEs != "ES") | !is.character(refOrEs)) {
        stop("Argument 'refOrEs' must read 'REF' (for reference data.frame) or 'ES' (for event sampling data.frame). Please check.")
    }
    
    
    # Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=RELEVANTINFO_ES,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=RELEVANTVN_REF)
    

    if ( (dateFormat != "ymd" & dateFormat != "mdy" & dateFormat != "dmy") | !is.character(dateFormat)) {
        stop("The date-format is invalid. Please choose from the 3 valid formats 'ymd', 'mdy', or 'dmy'. Type 'dateTimeFormats()' for help.")
    }

    if ( (timeFormat != "HMS" & timeFormat != "HM") | !is.character(timeFormat)) {
        stop("The time-format is invalid. Please choose from the 2 valid formats 'HMS' or 'HM'. Type 'dateTimeFormats()' for help.")
    }

    if(refOrEs == "REF") {

        if(!is.data.frame(refOrEsDf)) {
            stop("The argument 'refOrEsDf' must be of type data.frame.")
        }
        # ERROR handling end ---------------------------------------------

        dateTimeStartDf <- genDateTimeSingle(refOrEsDf, refOrEs=refOrEs, dateFormat = dateFormat, timeFormat = timeFormat, RELEVANTVN_ES = RELEVANTVN_ES, RELEVANTVN_REF = RELEVANTVN_REF)
        
        RELEVANTVN_REF_Temp <- dateTimeStartDf[["extendedVariableNameList"]]
        
        dateTimeEndDf <- genDateTimeSingle(refOrEsDf, refOrEs=refOrEs, dateFormat = dateFormat, timeFormat = timeFormat, startOrEnd="END", RELEVANTVN_ES = RELEVANTVN_ES, RELEVANTVN_REF = RELEVANTVN_REF_Temp)

        # # df_dateTime <- merge(dateTimeStartDf, dateTimeEndDf, all=TRUE)
        df_dateTime <- merge(dateTimeStartDf[["refOrEsDfSingle"]], dateTimeEndDf[["refOrEsDfSingle"]], all=TRUE)

        # Return the dataset
        list(refOrEsDf = df_dateTime, extendedVNList=dateTimeEndDf[["extendedVariableNameList"]])

    } else {
    	
    		if(!is.list(refOrEsDf) & all(sapply(refOrEsDf, is.data.frame))) {
			stop("The argument 'refOrEsDf' must be of type list, each element of the list must be of type data.frame.")
		}
        # ERROR handling end ---------------------------------------------

        esList <- list()
        if(!is.null(names(refOrEsDf))){
            esListNames <- names(refOrEsDf)
        }
        for(i in 1:length(refOrEsDf)) {
            dateTimeStartDf <- genDateTimeSingle(refOrEsDf[[i]], refOrEs=refOrEs, dateFormat = dateFormat, timeFormat = timeFormat, RELEVANTVN_ES = RELEVANTVN_ES, RELEVANTVN_REF = RELEVANTVN_REF)
            
            RELEVANTVN_ES_Temp <- dateTimeStartDf[["extendedVariableNameList"]]
            
            dateTimeEndDf <- genDateTimeSingle(refOrEsDf[[i]], refOrEs=refOrEs, dateFormat = dateFormat, timeFormat = timeFormat, startOrEnd="END", RELEVANTVN_ES = RELEVANTVN_ES_Temp, RELEVANTVN_REF = RELEVANTVN_REF)

            df_dateTime <- merge(dateTimeStartDf[["refOrEsDfSingle"]], dateTimeEndDf[["refOrEsDfSingle"]], all=TRUE)
            esList[[esListNames[i]]] <- df_dateTime
        }
        # esList
        list(refOrEsDf = esList, extendedVNList=dateTimeEndDf[["extendedVariableNameList"]])
    }    
}
