% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clean_labels.R
\name{clean_labels}
\alias{clean_labels}
\title{Standardise labels}
\usage{
clean_labels(
  x,
  sep = "_",
  transformation = "Any-Latin; Latin-ASCII",
  protect = ""
)
}
\arguments{
\item{x}{A vector of labels, normally provided as characters.}

\item{sep}{A character string used as separator, defaulting to '_'.}

\item{transformation}{a string to be passed on to \code{\link[stringi:stri_trans_general]{stringi::stri_trans_general()}}
for conversion. Default is "Any-Latin; Latin-ASCII", which will convert
any non-latin characters to latin and then converts all accented
characters to ASCII characters. See \code{\link[stringi:stri_trans_list]{stringi::stri_trans_list()}} for a
full list of options.}

\item{protect}{a character string defining the punctuation that should be
protected. This helps prevent meaninful symbols like > and < from being
removed.}
}
\description{
This function standardises labels e.g. used as variable names or character
string values, removing non-ascii characters, replacing diacritics (e.g. é,
ô) with their closest ascii equivalents, and standardises separating
characters. See details for more information on label transformation.\cr
}
\details{
The following changes are performed:
\itemize{
\item all non-ascii characters are removed
\item all diacritics are replaced with their non-accentuated equivalents,
e.g. 'é', 'ê' and 'è' become 'e'.
\item all characters are set to lower case
\item separators are standardised to the use of a single character provided
in \code{sep} (defaults to '_'); heading and trailing separators are removed.
}
}
\note{
Because of differences between the underlying transliteration engine
(ICU), the default transformations will not transilierate German umlaute
correctly. You can add them by specifying "de-ASCII" in the \code{transformation}
string after "Any-Latin".
}
\examples{
\dontrun{
clean_labels("-_-This is; A    WeÏrD**./sêntënce...")
clean_labels("-_-This is; A    WeÏrD**./sêntënce...", sep = ".")
input <- c("Peter and stëven",
           "peter-and.stëven",
           "pëtêr and stëven  _-")
input
clean_labels(input)

# Don't transliterate non-latin words
clean_labels(input, transformation = "Latin-ASCII")

# protect useful symbols
clean_labels(c("energy > 9000", "energy < 9000"), protect = "><")

# if you only want to clean accents, transform to lower, and transliterate,
# you can specify "[:punct:][:space:]" for protect:
clean_labels(input, protect = "[:punct:][:space:]")

# appropriately transliterate Germanic umlaute
if (stringi::stri_info()$ICU.system) {
  # This will only be true if you have the correct version of ICU installed

  clean_labels("'é', 'ê' and 'è' become 'e', 'ö' becomes 'oe', etc.", 
               transformation = "Any-Latin; de-ASCII; Latin-ASCII")
}
}
}
\author{
Thibaut Jombart \email{thibautjombart@gmail.com}, Zhian N. Kamvar
}
