#' Feldt's Coefficient
#'
#' @description
#' Compute Feldt's coefficient as an estimate of internal consistency
#' reliability.
#'
#' @param x A data frame or matrix containing item responses, with rows as
#'   subjects and columns as items.
#'
#'
#' @return A named list with:
#' \describe{
#'   \item{feldt}{Feldt's coefficient.}
#' }
#'
#'
#' @examples
#' data(data.u)
#' feldt(data.u)
#'
#' @export
feldt <- function(x) {
  # basic checks ---------------------------------------------------------------
  if (is.null(x)) {
    stop("`x` must not be NULL.")
  }
  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  # remove missing values (listwise), ensure numeric matrix --------------------
  x <- stats::na.exclude(x)
  x <- as.matrix(x)
  if (!is.numeric(x)) {
    stop("`x` must contain only numeric values.")
  }

  if (ncol(x) < 2L) {
    stop("`x` must have at least 2 items (columns) to compute Feldt's coefficient.")
  }

  # covariance matrix ----------------------------------------------------------
  cov_mat <- stats::cov(x)

  # sum of total variance
  vt <- sum(cov_mat)

  # sum of variance of each item
  vi <- sum(diag(cov_mat))

  # sum of covariance term used in Feldt's formula
  cov_sum <- sum(rowSums(cov_mat)^2)

  # guard against degenerate cases --------------------------------------------
  if (isTRUE(all.equal(vt^2 - cov_sum, 0))) {
    stop("Denominator in Feldt's coefficient is zero; coefficient is undefined.")
  }

  # Feldt's coefficient --------------------------------------------------------
  feldt <- vt * (vt - vi) / (vt^2 - cov_sum)     # EM4
  # feldt_alt <- vt^2 / (vt^2 - cov_sum) * (vt - vi) / vt  # Lee notes (equivalent)

  return(list(feldt = feldt))
}
