% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dpa.R
\name{dpa}
\alias{dpa}
\title{Dynamic Path Analysis}
\usage{
dpa(
  out.formula,
  mediator.formulas,
  id,
  data,
  boot.n = 200,
  method = "timereg",
  progress_bar = FALSE,
  ...
)
}
\arguments{
\item{out.formula}{Survival formula for Aalen's additive hazards model.}

\item{mediator.formulas}{Mediator regression formula (in case of a single mediator), or a list of regression formulas (in case of multiple mediators).
The formulas must be ordered according to Directed Acyclic Graph Structure (see Details).}

\item{id}{character string indicating which column of 'data' corresponds to the subject ID. Bootstrapping will be performed on this id.}

\item{data}{Data set in counting process format. In particular the data should contain a "start", "stop" and "event" column along with
any mediators and baseline covariates.}

\item{boot.n}{Number of bootstrap samples.}

\item{method}{The underlying implementation of Aalen's additive regression model. Defaults to "timereg", which relies on the timereg::aalen() implementation,
while method = "aareg" uses the survival::aareg() implementation.}

\item{progress_bar}{Boolean. If TRUE, show progress bar. Defaults to FALSE.}

\item{...}{other parameters passed to the Aalen's additive hazards model implementation. If method = "timereg", then ... will be passed to
timereg::aalen(), while if method = "aareg", then ... will be passed to survival::aareg(). If ... contains parameters that don't belong to the formalArgs of
the corresponding implementation then those parameters will be ignored.}
}
\value{
Object of class `dpa` with following fields:
\describe{
  \item{coefs}{list of estimated coefficients from each of the regressions listed in out.formula and mediator.formulas.}
  \item{boot.coefs}{list of bootstrap estimates corresponding to coefs. This stores all the bootstrap estimates to facilitate
calculation of direct, indirect and total effects along with bootstrap confidence intervals.}
  \item{meta}{a list keeping track of responses and covariates of each of the out.formula and mediator.formulas. Also keeps
track of all variable types and level names in case of factors.}
  \item{aalen}{Object storing information pertaining to the Aalen's additive model fit. Object is of class "aalen"
  if method="timereg", and of class "aareg" if method="aareg".}
}
}
\description{
Dynamic Path Analysis
}
\details{
\code{dpa} performs Dynamic Path Analysis of a Directed Acyclic Graph (DAG). The out.formula
can have as covariates all mediators listed in mediator.formulas. The mediator.formulas must obey the
following DAG structure rule: The response of the k-th formula cannot appear as covariate in any of the formulas
k+1, ..., length(mediator.formulas).
}
\examples{
library(dpasurv)

data(simdata)

set.seed(1)

# Perform dynamic path analysis:
# We set boot.n=30 for the example to run fast, should be set large enough
# so that results don't change meaningfully for different seeds.
s <- dpa(Surv(start,stop,event)~M+x, list(M~x), id="subject", data=simdata, boot.n=30)

# Calculate cumulative direct, indirect, and total effects:
direct <- effect(x ~ outcome, s)
indirect <- effect(x ~ M ~ outcome, s)
total <- sum(direct, indirect)

# Plot the effects using basic graphics:
layout1x3 <- par(mfrow=c(1,3))
plot(direct); abline(h=0, lty=2, col=2)
plot(indirect); abline(h=0, lty=2, col=2)
plot(total); abline(h=0, lty=2, col=2)

# restore user's graphical parameters:
par(layout1x3)

# Plot the effects using ggplot2 graphics:
ggplot.effect(list(direct, indirect, total))

}
