% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample.R
\name{post_sample}
\alias{post_sample}
\alias{post_sample.gp}
\alias{post_sample.gpvec}
\alias{post_sample.dgp2}
\alias{post_sample.dgp2vec}
\alias{post_sample.dgp3}
\alias{post_sample.dgp3vec}
\title{Generates joint posterior samples from a trained GP/DGP}
\usage{
post_sample(object, x_new, nper = 1, ...)

\method{post_sample}{gp}(object, x_new, nper = 1, grad = FALSE, cores = 1, ...)

\method{post_sample}{gpvec}(
  object,
  x_new,
  nper = 1,
  m = NULL,
  ord_new = NULL,
  grad = FALSE,
  cores = 1,
  ...
)

\method{post_sample}{dgp2}(
  object,
  x_new,
  nper = 1,
  grad = FALSE,
  mean_map = TRUE,
  cores = 1,
  ...
)

\method{post_sample}{dgp2vec}(
  object,
  x_new,
  nper = 1,
  m = NULL,
  ord_new = NULL,
  grad = FALSE,
  mean_map = TRUE,
  cores = 1,
  ...
)

\method{post_sample}{dgp3}(object, x_new, nper = 1, mean_map = TRUE, cores = 1, ...)

\method{post_sample}{dgp3vec}(
  object,
  x_new,
  nper = 1,
  m = NULL,
  ord_new = NULL,
  mean_map = TRUE,
  cores = 1,
  ...
)
}
\arguments{
\item{object}{object from \code{fit_one_layer}, \code{fit_two_layer}, or 
\code{fit_three_layer} with burn-in already removed}

\item{x_new}{vector or matrix of predictive input locations}

\item{nper}{the number of samples to generate from each MCMC iteration.  The
total number of samples will equal \code{nper*object$nmcmc}}

\item{...}{N/A}

\item{grad}{logical indicating whether to additionally calculate/return 
samples of the gradient (one and two layer models only)}

\item{cores}{number of cores to use for SNOW parallelization}

\item{m}{size of Vecchia conditioning sets (only for fits with 
\code{vecchia = TRUE}), defaults to the twice the \code{m} used for MCMC}

\item{ord_new}{optional ordering for Vecchia approximation with \code{lite = FALSE}, 
must correspond to rows of \code{x_new}, defaults to random, is 
applied to all layers in deeper models}

\item{mean_map}{logical indicating whether to map hidden layers using 
conditional mean (\code{mean_map = TRUE}) or using a random sample
from the full MVN distribution (two or three layer models only)}
}
\value{
If \code{grad = FALSE}, returns matrix of samples.  Rows correspond to 
        \code{x_new} locations.  If \code{grad = TRUE}, returns a list with 
        \code{y} and \code{dydx} containing the respective samples.
}
\description{
Acts on a \code{gp}, \code{gpvec}, \code{dgp2}, \code{dgp2vec},
    \code{dgp3}, or \code{dgp3vec} object.  Generates joint samples from the 
    posterior distribution at the provided locations.
}
\details{
By default, one sample is generated per each MCMC iteration.  This
    may be increased with the \code{nper} argument.
    
    SNOW parallelization reduces computation time but requires 
    more memory storage.
}
\examples{
\donttest{
# Simple step function
f <- function(x) {
  return(pnorm((x - 0.5) / 0.065))
}

# Training data
x <- seq(0, 1, length = 5)
y <- f(x)

# Testing data
xx <- seq(0, 1, length = 100)
yy <- f(xx)

plot(xx, yy, type = "l")
points(x, y, col = 2)

# Conduct MCMC
fit <- fit_two_layer(x, y, nmcmc = 2000, true_g = 1e-6, cov = "exp2")
plot(fit, hidden = TRUE)
fit <- trim(fit, 1000, 2)

# Generate posterior samples, including gradients
samples <- post_sample(fit, xx, grad = TRUE, cores = 1)

# Plot samples
par(mfrow = c(1, 2))
matplot(xx, t(samples$y), type = "l")
points(x, y, pch = 20)
matplot(xx, t(samples$dy), type = "l")
}

}
\references{
Sauer, A. (2023). Deep Gaussian process surrogates for computer experiments. 
     *Ph.D. Dissertation, Department of Statistics, Virginia Polytechnic Institute and State University.*
     \cr\cr
}
