% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/credsubs.R
\name{credsubs.level}
\alias{credsubs.level}
\title{Compute the maximum credible levels at which conclusions may be drawn}
\usage{
credsubs.level(
  params,
  design = NULL,
  FUN = function(x, params) {     params \%*\% t(x) },
  threshold = 0,
  method = c("asymptotic", "quantile"),
  step.down = TRUE,
  sides = c("both", "exclusive", "inclusive"),
  est.FUN = mean,
  var.FUN = sd,
  point.estimate = NULL,
  track = numeric(0),
  verbose = FALSE,
  z.store = c("ram", "recompute", "disk")
)
}
\arguments{
\item{params}{A numeric matrix whose rows are draws from the posterior
distribution of either the regression surface or the
parameter vector.}

\item{design}{(Optional) A numeric matrix whose rows are covariate points
over which the band is to be constructed.}

\item{FUN}{(Optional) a function of the form \code{function(x, params)}
that takes a row of \code{design} and the entire \code{params}
matrix and returns a vector of the same length of \code{x}
representing the regression surface.}

\item{threshold}{Numeric; the value of \code{FUN} above which
a covariate is included in the target subset.}

\item{method}{Either "asymptotic" (default) or "quantile"; see details.}

\item{step.down}{Logical (default \code{TRUE}); should the step-down
procedure be used?}

\item{sides}{One of "both" (default), "exclusive", or "inclusive".
Which bounds should be constructed?}

\item{est.FUN}{The function used to produce estimates of the regression
surface. Default is \code{mean}.}

\item{var.FUN}{The function used to quantify the variability of the
regression surface posterior. Default is \code{sd}.}

\item{point.estimate}{If not null, replaces the mean and sets the reference 
around which the standard error is computed.
Useful for bootstrapping methods.
Treated as a row of the \code{params} matrix.}

\item{track}{A numeric vector of indices indicating which rows (default none)
of the design matrix should have the sample of the corresponding
\code{FUN(x, params)} returned.}

\item{verbose}{Logical (default \code{FALSE}); print progress?}

\item{z.store}{How should certain intermediate computations be handled?
See details.}
}
\value{
An object of class \code{credsubs.level}, which contains:
\describe{
  \item{\code{level}}{A numeric vector indicating the maximum credible
                      level at which a conclusion may be drawn at each
                      covariate point.}
  \item{\code{sign}}{A numeric vector indicating the which credible subsets
                     of which each covariate point is a member at the
                     credible level indicated by \code{level}. Exclusive
                     and inclusive: 1, inclusive only: 0, neither: -1.}
  \item{\code{threshold}}{As provided.}
  \item{\code{method}}{As provided.}
  \item{\code{step.down}}{As provided.}
  \item{\code{sides}}{As provided.}
  \item{\code{est}}{Posterior estimate of the regression surface.}
  \item{\code{est.FUN}}{As provided.}
  \item{\code{var}}{Summary of posterior variability of the regression
                    surface.}
  \item{\code{var.FUN}}{As provided.}
  \item{\code{trace}}{The posterior samples of the regression surface
                      indicated by the \code{track} argument.}
}
}
\description{
For each covariate point, \code{credsubs.level} computes the maximum
credible level at which a conclusion may be drawn at each point, and
what that conclusion is.
}
\details{
If design is NULL (default), it is taken to be the identity
matrix of dimension ncol(params), so that the rows of params
are treated as draws from the posterior FUN(x, params).

The 'asymptotic' method assumes that the marginal posteriors of 
the FUN(x, params) are asymptotically normal and is usually
significantly faster and less memory-intensive than the 'quantile'
method, which makes no such assumption.

By default (\code{z.store = "ram"}), the maximum credible level computation
stores a potentially very large amount of intermediate computation results
in memory. If not enough memory is available, \code{z.store = "disk"}
uses the \code{ff} package to store the intermediate results on disk,
which can still be fairly quick if the storage is fast (e.g. a local SSD).
Alternatively, \code{z.store = "recompute"} discards the intermediate
results and recomputes whenever needed. This uses minimal memory, but
is usually the slowest option.
}
\examples{
### Sample from regression surface posterior
reg.surf.sample <- matrix(rnorm(1000, mean=1:10), ncol=2, byrow=TRUE)
credsubs.level(reg.surf.sample)

### Parametric case
design <- cbind(1, 1:10)
params <- matrix(rnorm(200, mean=1:2), ncol=2, byrow=TRUE)
credsubs(params, design)

### With custom function
params.sd <- cbind(1 / rgamma(100, 1), params)
FUN.sd <- function(x, params) { params[, -1] \%*\% t(x) / params[, 1] }
credsubs(params.sd, design, FUN.sd, threshold=1)

}
