# R/museum.R
# Political Linguistics Museum functions

#' List all gaffes in the museum
#'
#' @return Character vector of gaffe IDs
#' @export
#' @examples
#' museum_list()
museum_list <- function() {
  gaffes$id
}

#' Look up a specific gaffe
#'
#' @param id Character. The gaffe ID (e.g., "covfefe", "potatoe")
#' @return List with gaffe details
#' @export
#' @examples
#' museum_lookup("covfefe")
museum_lookup <- function(id) {
  idx <- which(gaffes$id == id)
  if (length(idx) == 0) {
    stop(sprintf("Gaffe '%s' not found in museum. Use museum_list() to see available gaffes.", id))
  }
  as.list(gaffes[idx, ])
}

#' Get a random gaffe from the museum
#'
#' @return List with gaffe details
#' @export
#' @examples
#' set.seed(42)
#' museum_random()
museum_random <- function() {
  idx <- sample(nrow(gaffes), 1)
  as.list(gaffes[idx, ])
}

#' Filter gaffes by person
#'
#' @param pattern Character. Regex pattern to match person name
#' @return Data frame of matching gaffes
#' @export
#' @examples
#' museum_by_person("Trump")
museum_by_person <- function(pattern) {
  gaffes[grepl(pattern, gaffes$person, ignore.case = TRUE), ]
}

#' Filter gaffes by year
#'
#' @param year Integer. Year to filter by
#' @return Data frame of matching gaffes
#' @export
#' @examples
#' museum_by_year(2017)
museum_by_year <- function(year) {
  gaffes[gaffes$year == year, ]
}

#' Filter gaffes by typo type
#'
#' @param type Character. Type of typo (e.g., "blend", "keyboard_mash")
#' @return Data frame of matching gaffes
#' @export
#' @examples
#' museum_by_type("blend")
museum_by_type <- function(type) {
  gaffes[gaffes$typo_type == type, ]
}
