% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matcalc.R
\name{illcond_sample}
\alias{illcond_sample}
\title{Ill-conditioned sampling}
\usage{
illcond_sample(A, bmax = 100, Dbmax = 1, ncyc = 1e+05, iseed = NULL)
}
\arguments{
\item{A}{The \eqn{n \times n} matrix of coefficients of the 
unknowns in the linear system.}

\item{bmax}{A numeric number providing the interval, (0,bmax),
in which the \eqn{n} uniformly random components of
\eqn{b} are selected. Default value is 100.}

\item{Dbmax}{A numeric number providing the interval, (0,Dbmax),
in which the \eqn{n} uniformly random components of
\eqn{\Delta b} are selected. Default value is 1.}

\item{ncyc}{An integer indicating the number of uniform random
selection of the \eqn{n} components of \eqn{b} and the
\eqn{n} components of \eqn{\Delta b}. The higher this
number, the higher the chance of getting a high 
relative solution number, but the longer the execution
time of the function. Default is 100000.}

\item{iseed}{An integer. The seed starting random generation. If
a value is provided, the (pseudo-)random generation 
will reproduce exactly the same \eqn{b} and
\eqn{\Delta b}. Default is NULL, which means that the
seed will be randomly chosen at every execution of 
the function.}
}
\value{
A named list with names b, a vector equal of the right-hand
        side of the linear system, and Db, a vector equal to the
        perturbations, \eqn{\Delta b}, to be applied to \eqn{b}.
}
\description{
Random sampling, based on the uniform distribution, of the 
right-hand side, \eqn{b}, of a linear system and of a perturbation, 
\eqn{\Delta b}, so that the solution of \eqn{Ax=b} is very 
different from the solution of \eqn{Ax=b+\Delta b}.
}
\details{
The degree of ill-conditioning of a system is not only measured 
by the matrix's condition number, but also from the solution
relative error. If \eqn{\Delta x} is the difference between the
solution, \eqn{x}, of the system related to \eqn{b} and the
solution, \eqn{x'}, of the system related to \eqn{b'= b-\Delta b},
then the ratio of the norm of \eqn{\Delta x} and the norm of
\eqn{x}, is the solution relative error. Norms are Frobenius norms. 
This function returns a named list with b and Db the chosen 
\eqn{b} and \eqn{\Delta b}, based on random sampling of a 
specified region.
}
\examples{
# This is a simple but ill-conditioned matrix
A <- matrix(c(2,1,1.99,1),ncol=2)

# Select b and Db randomly, starting with iseed=2341
ltmp <- illcond_sample(A,iseed=2341)
names(ltmp)

# b and b'
b <- ltmp$b
Db <- ltmp$Db
b2 <- b-Db

# Solution for b
x <- solve(A,b)
print(x)

# Solution for b'
x2 <- solve(A,b2)
print(x2)

# Difference
Dx <- x-x2

# Solution relative error (Frobenius norm)
print(norm(Dx,"F")/norm(x,"F"))

# Upper limit
Ainv <- solve(A)
print(norm(A,"F")*norm(Ainv,"F")*norm(Db,"F")/norm(b,"F"))

}
