% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/categorize.R
\name{categorize}
\alias{categorize}
\alias{categorize.data.frame}
\alias{categorize.tbl_df}
\alias{categorize.data.table}
\alias{categorize.codified}
\title{Categorize cases based on external data and classification scheme}
\usage{
categorize(x, ...)

\method{categorize}{data.frame}(x, ...)

\method{categorize}{tbl_df}(x, ...)

\method{categorize}{data.table}(x, ..., codedata, id, code, codify_args = list())

\method{categorize}{codified}(
  x,
  ...,
  cc,
  index = NULL,
  cc_args = list(),
  check.names = TRUE,
  .data_cols = NULL
)
}
\arguments{
\item{x}{data set with mandatory character id column
(identified by argument \code{id = "<col_name>"}),
and optional \code{\link{Date}}  of interest
(identified by argument \code{date = "<col_name>"}).
Alternatively, the output from \code{\link[=codify]{codify()}}}

\item{...}{arguments passed between methods}

\item{codedata}{external code data with mandatory character id column
(identified by \code{id = "<col_name>"}),
code column (identified by argument \code{code = "<col_name>"})
and optional \code{\link{Date}} column
(identified by \code{codify_args = list(code_date = "<col_name>")}).}

\item{id}{name of unique character id column found in
both \code{x}and \code{codedata}.
(where it must not be unique).}

\item{code}{name of code column in \code{codedata}.}

\item{codify_args}{Lists of named arguments passed to \code{\link[=codify]{codify()}}}

\item{cc}{\code{\link{classcodes}} object (or name of a default object from
\code{\link[=all_classcodes]{all_classcodes()}}).}

\item{index}{Argument passed to \code{\link[=index]{index()}}.
A character vector of names of columns with index weights from the
corresponding classcodes object (as supplied by the \code{cc}argument).
See \code{attr(cc, "indices")} for available options.
Set to \code{FALSE} if no index should be calculated.
If \code{NULL}, the default, all available indices (from \code{attr(cc, "indices")})
are provided.}

\item{cc_args}{List with named arguments passed to
\code{\link[=set_classcodes]{set_classcodes()}}}

\item{check.names}{Column names are based on \code{cc$group}, which might include
spaces. Those names are changed to syntactically correct names by
\code{check.names = TRUE}. Syntactically invalid, but grammatically correct
names might be preferred for presentation of the data as achieved by
\code{check.names = FALSE}. Alternatively, if \code{categorize} is called repeatedly,
longer informative names might be created by
\code{cc_args = list(tech_names = TRUE)}.}

\item{.data_cols}{used internally}
}
\value{
Object of the same class as \code{x} with additional logical columns
indicating membership of groups identified by the
\code{classcodes} object (the \code{cc} argument).
Numeric indices are also included if requested by the \code{index} argument.
}
\description{
This is the main function of the package, which relies of a triad of objects:
(1) \code{data} with unit id:s and possible dates of interest;
(2) \code{codedata} for corresponding
units and with optional dates of interest and;
(3) a classification scheme (\code{\link{classcodes}} object; \code{cc}) with regular
expressions to identify and categorize relevant codes.
The function combines the three underlying steps performed by
\code{\link[=codify]{codify()}}, \code{\link[=classify]{classify()}} and \code{\link[=index]{index()}}.
Relevant arguments are passed to those functions by
\code{codify_args} and \code{cc_args}.
}
\examples{
# For this example, 1 core would suffice:
old_threads <- data.table::getDTthreads()
data.table::setDTthreads(1)

# For some patient data (ex_people) and related hospital visit code data
# with ICD 10-codes (ex_icd10), add the Elixhauser comorbidity
# conditions based on all registered ICD10-codes
categorize(
   x            = ex_people,
   codedata     = ex_icd10,
   cc           = "elixhauser",
   id           = "name",
   code         = "icd10"
)


# Add Charlson categories and two versions of a calculated index
# ("quan_original" and "quan_updated").
categorize(
   x            = ex_people,
   codedata     = ex_icd10,
   cc           = "charlson",
   id           = "name",
   code         = "icd10",
   index        = c("quan_original", "quan_updated")
)


# Only include recent hospital visits within 30 days before surgery,
categorize(
   x            = ex_people,
   codedata     = ex_icd10,
   cc           = "charlson",
   id           = "name",
   code         = "icd10",
   index        = c("quan_original", "quan_updated"),
   codify_args  = list(
      date      = "surgery",
      days      = c(-30, -1),
      code_date = "admission"
   )
)



# Multiple versions -------------------------------------------------------

# We can compare categorization by according to Quan et al. (2005); "icd10",
# and Armitage et al. (2010); "icd10_rcs" (see `?charlson`)
# Note the use of `tech_names = TRUE` to distinguish the column names from the
# two versions.

# We first specify some common settings ...
ind <- c("quan_original", "quan_updated")
cd  <- list(date = "surgery", days = c(-30, -1), code_date = "admission")

# ... we then categorize once with "icd10" as the default regular expression ...
categorize(
   x            = ex_people,
   codedata     = ex_icd10,
   cc           = "charlson",
   id           = "name",
   code         = "icd10",
   index        = ind,
   codify_args  = cd,
   cc_args      = list(tech_names = TRUE)
) \%>\%

# .. and once more with `regex = "icd10_rcs"`
categorize(
   codedata     = ex_icd10,
   cc           = "charlson",
   id           = "name",
   code         = "icd10",
   index        = ind,
   codify_args  = cd,
   cc_args      = list(regex = "icd10_rcs", tech_names = TRUE)
)



# column names ------------------------------------------------------------

# Default column names are based on row names from corresponding classcodes
# object but are modified to be syntactically correct.
default <-
   categorize(ex_people, codedata = ex_icd10, cc = "elixhauser",
              id = "name", code = "icd10")

# Set `check.names = FALSE` to retain original names:
original <-
  categorize(
    ex_people, codedata = ex_icd10, cc = "elixhauser",
    id = "name", code = "icd10",
    check.names = FALSE
   )

# Or use `tech_names = TRUE` for informative but long names (use case above)
tech <-
  categorize(ex_people, codedata = ex_icd10, cc = "elixhauser",
    id = "name", code = "icd10",
    cc_args = list(tech_names = TRUE)
  )

# Compare
tibble::tibble(names(default), names(original), names(tech))

# Go back to original number of threads
data.table::setDTthreads(old_threads)
}
\seealso{
Other verbs: 
\code{\link{classify}()},
\code{\link{codify}()},
\code{\link{index_fun}}
}
\concept{verbs}
