#' @name findATE2
#' @title Find the cutoff values of biomarkers based on the average subpopulation treatment effect
#'
#' @param z2interval a numeric vector of two values, including the lower and upper limits of the initial interval for z2
#' @param kkk the researchers' weighting preference between the two biomarkers
#' @param muminusmu0 a number of the difference between the mean of outcome and the minimal clinically important treatment effect
#' @param kappa a number of the correlation coefficient between two biomarkers
#' @param rhovec a numeric vector of two correlation coefficients between the output and two biomarkers
#' @param sigma a number of the standard deviation of outcome
#' @param cDel the desired average subpopulation treatment effect
#'
#' @returns a numeric vector of two values which are the cutoff values for z1 and z2
#' @export
#'
#' @author Jiangtao Gou
#' @author Fengqing Zhang
#' @references
#' Zhang, F. and Gou, J. (2025). Using multiple biomarkers for patient enrichment in two-stage clinical designs. Technical Report.
#'
#' @examples
#' z2interval <- c(-5, 5)
#' kkk <- 1
#' muminusmu0 <- 1.8
#' kappa <- 0.1
#' rhovec <- c(0.1, 0.2)
#' sigma <- 1
#' cDel <- 2.5
#' findATE2(z2interval, kkk, muminusmu0, kappa, rhovec, sigma, cDel)
findATE2 <- function (z2interval, kkk, muminusmu0, kappa, rhovec, sigma, cDel) {
  if (kkk >= 1) {
    minZ2 <- stats::qnorm(p = 1/kkk, lower.tail = FALSE)
    if (minZ2 > z2interval[1]) {
      stop("Please adjust the lower limit of the initial interval for z2.")
    }
  }
  result <- stats::uniroot(f = targetDel, interval = z2interval,
                    tol = .Machine$double.eps^0.8, maxiter = 1000,
                    kkk = kkk, muminusmu0 = muminusmu0, kappa = kappa, rhovec = rhovec, sigma = sigma, cDel = cDel)
  myz2 <- result$root
  myz1 <- stats::qnorm(p = kkk * stats::pnorm(q = myz2, lower.tail = FALSE), lower.tail = FALSE)
  cutoff <- c(myz1, myz2)
  return(cutoff)
}
