##
# ------------------------------------------------------------------------
#
# "mark_boot(X,func,B,...)" --
#
# Bootstrap trajectory of the Markov chain using the Nadaraya-Watson estimator
#
# ------------------------------------------------------------------------
##
#' @aliases mark_boot
#' @title Bootstraping Markov chain
#' @description The function fo bootstrap the Markov chain using estimator of the 
#' transition kernel
#' @param X A numeric vector representing a Markov chain.
#' @param func The function to apply to each sample.
#' @param B A positive integer; the number of bootstrap samples.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details  The method is based on estimating the transition
#'   kernel of the chain,
#'  which is used to generate the bootstrap time series.
#'  The transition density is estimated using some Gaussian kernel.
#'  
#' @return Returns an object of class \code{boodd}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted..
#' 
#' Prakasa Rao, B. L. S. and Kulperger, R. J. (1989). Bootstrapping a
#' finite state Markov chain. \emph{Sankhya - Series A}, \bold{51}, 178-191.
#' 
#' Rajarshi, M.B. (1990). Bootstrap in Markov-Sequences Based on Estimates of
#' Transition Density. \emph{Annals of the Institute of Statistical Mathematics},
#' \bold{42}, 253-268.
#' @seealso 
#' \code{\link{blockboot}},
#' \code{\link{regenboot}},
#' \code{\link{findBestEpsilon}}.
#' @keywords "Markov chains" "Markov bootstrap" "Transition density estimators"
#' @export
#' @examples 
#' set.seed(12345)
#' phi=0.6
#' n=200
#' X <- arima.sim(list(order=c(1,0,0),ar=phi),n=n)
#'  boo1=mark_boot(X,mean,199)
#'  boot_dist(boo1)
#'  # Compute confidence intervals
#'  confint(boo1,method="all")
##
mark_boot <- function(X,func,B,...) {
  
  # Test the value returned by func
  y <- func(X,...)
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  res <- matrix(nrow=B,ncol=len)
  if (is.matrix(X)) {
    stop("X should be a univariate time series ")
  } else {
    n <- length(X)
    h=n^(-1/5) # optimal bandwidth
  }
  
  # Bootstrap loop : B resamplings 
  
  for (b in 1:B) {
    
    # Initialize Bootstrap Markov based time series 
    Xstar=rep(0,n)
    
    # Pick a starting point in kernel density estimator h_hat
    
    X0=X[sample(1:n,size=1)]+h*rnorm(1)
    
    for(i in 1:n){
      weight=dnorm((X0-X)/h)/(n*h*f_hat(X0,X))
      ind=sum((cumsum(weight)-runif(1)<0))+1
      if (ind < n) Xstar[i]=X[ind]+h*rnorm(1)
      else Xstar[i]=X[n]
      X0=Xstar[i]}
    
    res[b,] <- func(Xstar,...) 
  }
  # end of loop 
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- "iid"
  attr(obj,"func") <- func
  return(obj)
}

