## ------------------------------------------------------------------------
# 
# "boot_res(X,n,h,kernel,t)" 
# 
# TFT Residual bootstrap
#
# ------------------------------------------------------------------------
##


#' @aliases boot_res
#' @title TFT Residual Bootstrap.
#' @description The residual bootstrap is used to bootstrap the Fourier coefficients for 
#' the Time Frequency Toggle (TFT)-Bootstrap (see Kirch and Politis (2011)).
#' @param X A numeric vector representing a time series.
#' @param n An integer; by default is the length of time series \code{X} but allow for a 
#' smaller sample size m to perform moon bootstrap.
#' @param h A positive numeric value specifying the bandwidth used to compute the kernel estimator 
#' in case of local bootstrap. 
#' By default it is equal to \eqn{n^{-2/3}}.
#' @param kernel An integer value indicating the kernel type. Use \code{0} 
#' for the Daniell kernel or any other value for the Bartlett-Priestley (Epanechnikov) kernel 
#' (by default).
#' @param t An integer indicating the number of bootstrap replications.
#' @details The function first centers \code{X} and calculates its Fourier coefficients.
#'  It then estimates 
#' the spectral density using a kernel density estimator with parameters kernel \code{kernel} and 
#' bandwidth \code{h}, which standardizes the residuals accordingly. These 
#' standardized residuals are resampled with replacement to create bootstrap replicates.
#' @return  A matrix where each column contains a bootstrap replicate of the time series \code{X}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Kirch, C. and Politis, D. N. (2011). TFT-Bootstrap: Resampling time series in the frequency
#'  domain to obtain replicates in the time domain, \emph{Annals of Statistics}, vol.
#' @author We are grateful to Claudia Kirch for providing the original code in \code{R}.
#' @seealso \code{\link{tft_boot}}, \code{\link{boot_local}},
#' \code{\link{boot_wild}}.
#' @keywords Bootstrap "Frequency and time domain" "Fast Fourier Transform" 
#' @export
#' @examples 
#' # see the mother function tft_boot


boot_res<-function(X,n=length(X),h,kernel,t){
  # Test if n is even /odd
  if (n %%2 ==1){ n=n-1
  X=X[2:n]
  }
  Z<-X-mean(X)
  fourier<-fft(Z)/sqrt(n)
  foucoef<-c(Re(fourier[2:(n/2)]),Im(fourier[2:(n/2)]))
  
  periodograms<-((Re(fourier[2:(n/2)]))^2+(Im(fourier[2:(n/2)]))^2)/(2*pi)
  sdens<-sdens_est(periodograms,kernel,n,h)
  
  #attention: if n and h are such that trunc(n*h)=1, then sdens_est[1]=NA
  sqrtfreq<-sqrt(pi)*c(sqrt(sdens),sqrt(sdens))
  err<-foucoef/(sqrtfreq)
  err<-(err-mean(err))/sqrt(var(err))
  
  erg<-array(rep(0,times=t*n),dim=c(n,t))
  
  N<-(n/2-1)
  for(i in 1:t){
    neu<-sample(err,replace=TRUE)
    neu<-sqrtfreq*neu
    neu2<-neu[1:N]+1i*neu[(N+1):(n-2)]
    
    erg[,i]<-Re(fft(c(0,neu2,0,Conj(neu2[N:1])),inverse=TRUE)/sqrt(n))
    # Notice that by construction Z is centered , see Kirch and Politis for bootstrapping the mean in this context
  }
  return(erg)
}
