#' Computes the $U$-estimator for integer trees per-are (1 ha=100ares)
#'
#' @details The original function was written by Dr Oscar García, and
#' the corresponding reference is provided. The current function has only some
#' small changes.
#'
#' @title Function to compute the U-estimator for a variable from a sample plot
#' @param y.by.sortx a vector having the tree-level variable of interest being
#' already sorted by a sorting-variable.
#' @param nare number of trees per are for the sample plot. Remember that
#' 1 are=100 m2 or 1 ha=100 ares. "nare" it is an alternative to express
#'  stand density in trees/ha, here instead the unit is "trees/are".
#'   nare=length(y.by.sortx)/(plot.area.ares).  If "nare" is not an integer, it
#'   is rounded to the nearest integer, with a warning.
#'
#' @return The main output is the U-estimator
#'
#' @author Dr Oscar García.
#' @references
#' - Garcia O, Batho A. 2005. Top height estimation in lodgepole pine
#' sample plots. Western Journal of Applied forestry 20(1):64-68.
#'
#' @examples
#'
#' #Creates a fake dataframe
#' h <- c(29.1,28, 24.5, 26, 21,20.5,20.1);
#' trees.per.plot<-35; plot.area.m2<-500;
#' exp.factor.ha<-10000/plot.area.m2;exp.factor.ha
#' #Remember 1 are= 100 m2 o 1 ha= 100 ares
#' plot.area.ares<-plot.area.m2/100; plot.area.ares
#' plot.area.ha<-plot.area.m2/10000;plot.area.ha
#' n.ha<-trees.per.plot/plot.area.ha;n.ha #*(10000/plot.area.m2)
#' n.are<-trees.per.plot/plot.area.ares;n.are
#' #Using the domvar function
#' uestimator(y.by.sortx=h,nare=n.are)
#' @rdname uestimator
#' @export
#@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
uestimator<- function(y.by.sortx = y.by.sortx, nare = nare){
  m <- round(nare)
  if (m != nare) warning("Trees/are = ", nare, " has been rounded")
  sum(choose(0:(length(y.by.sortx) - 1), m - 1) * y.by.sortx) / choose(length(y.by.sortx), m)
}

