# This function performs the two-sample scale hypothesis method with missing data
# proposed in the paper "Scale two-sample testing with arbitrarily missing data".
# The data are assumed to be univariate and distinct, i.e. no ties.
# When missing data is presented, this function returns a p-value that controls
# the Type I error regardless of the values of missing data,
# along with the bounds of the Ansari-Bradley test statistic,
# and the bounds of the p-value without missing data.
#' @md
#' @export
#'
#' @title Ansari-Bradley Test with Arbitrarily Missing Data
#'
#' @description Performs the two-sample Ansari-Bradley test for univariate, distinct
#' data in the presence of missing values with controlled Type I error.
#'
#' @usage abwm.test(X, Y, alternative = c("two.sided", "less", "greater"))
#'
#' @param X,Y numeric vectors of data values with potential missing data.
#' Inf and -Inf values will be omitted.
#' @param alternative a character string specifying the alternative hypothesis,
#' must be one of "two.sided" (default), "greater" or "less".
#' You can specify just the initial letter.
#'
#' @details \code{abwm.test()} performs the Ansari-Bradley scale testing method
#' (Ansari & Bradley, 1960) in the presence of missing data, as described in Zeng et al. (2025).
#' Suppose \eqn{X = \{x_{1}, \ldots, x_{n}\}} and \eqn{Y = \{y_{1}, \ldots, y_{m}\}},
#' the Ansari-Bradley test statistic in Zeng et al. (2025) is defined as
#' \deqn{\sum_{i = 1}^{n} \left|R(x_i, X \cup Y) - \frac{n + m + 1}{2}\right|,}
#' where \eqn{R(x_i, X \cup Y)} denotes the rank of \eqn{x_i} in the set
#' \eqn{X \cup Y}. The function \code{abwm.test()} returns the tight bounds of
#' the Ansari-Bradley test statistic and its p-value obtained using the normal approximation.
#' The p-value of the test method proposed in Zeng et al. (2025) is then
#' returned as the maximum possible p-value of the Ansari-Bradley test.
#' This method (Zeng et al., 2025) does not make any missing data assumptions, and
#' controls the Type I error regardless of the values
#' of missing data.
#'
#' @return
#'
#'  \item{p.value}{the p-value for the test.}
#'
#'  \item{bounds.statistic}{bounds of the value of the Ansari-Bradley
#'  test statistic.}
#'
#'  \item{bounds.pvalue}{bounds of the p-value of the Ansari-Bradley
#'  test.}
#'
#'  \item{alternative}{a character string describing the alternative hypothesis.}
#'
#'  \item{data.name}{a character string giving the names of the data.}
#'
#' @references
#' \itemize{
#'  \item Y. Zeng, N. M. Adams, D. A. Bodenham. Scale two-sample testing with
#'  arbitrarily missing data. arXiv preprint arXiv:2509.20332. 2025.
#'
#'  \item A. R. Ansari, R. A. Bradley. Rank-Sum Tests for Dispersions.
#'  \emph{The Annals of Mathematical Statistics}, pages 1174–1189, 1960.
#'
#' }
#'
#' @seealso [stats::ansari.test()] when data are completely observed.
#'
#' @examples
#' #### Assume all values are distinct.
#' X <- c(6.2, 3.5, NA, 7.6, 9.2)
#' Y <- c(0.2, 1.3, -0.5, -1.7)
#'
#' ## Perform the two-sided Ansari-Bradley testing with missing data.
#' abwm.test(X, Y, alternative = 'two.sided')
#'
#' ## Consider one-sided testing.
#' abwm.test(X, Y, alternative = 'less')


abwm.test <- function(X, Y, alternative = c("two.sided", "less",
                                            "greater"))
{
  alternative <- match.arg(alternative)
  DNAME <- paste(deparse(substitute(X)), "and", deparse(substitute(Y)))

  # Remove all infinite
  X <- X[is.finite(X) | is.na(X)]
  Y <- Y[is.finite(Y) | is.na(Y)]

  # check ties
  checkTies(X,Y)

  ########### compute bounds
  MINAB <- MinAnsariBradley(X,Y)
  MAXAB <- MaxAnsariBradley(X,Y)
  BOUNDSAB <- c(MINAB, MAXAB)
  BOUNDSPVALUE <- AnsariBradleyTestMissingData(X, Y, MINAB, MAXAB,
                                               alternative = alternative)
  RES <- list(p.value = BOUNDSPVALUE[2],
              bounds.statistic = BOUNDSAB,
              bounds.pvalue =  BOUNDSPVALUE,
              alternative = alternative,
              data.name = DNAME)
  return(RES)
}
