#' Mutual Information for Continuous Variables using kNN
#'
#' @param x Numeric vector.
#' @param y Numeric vector.
#' @param k Number of nearest neighbors to use; default is 5.
#' @param direct Logical; if \code{TRUE}, mutual information is calculated
#' using the k-nearest neighbors (kNN) estimator; if \code{FALSE}, it is computed
#' via entropy estimates as \eqn{I(X;Y) = H(X) + H(Y) - H(X,Y)}. Default is \code{TRUE}.
#'
#' @return Numeric; an estimate of the mutual information.
#' @export
#' @importFrom FNN mutinfo
#'
#' @examples
#' set.seed(123)
#' x <- rnorm(1000)
#' y <- rnorm(1000)
#' # Close to 0 if they are independent
#' mut_info_knn(x, y, k = 5)
#' y <- 100 * x + rnorm(length(x), 0, 12)
#' # Far from 0 if they are not independent
#' mut_info_knn(x, y, k = 5)
mut_info_knn <- function(x, y, k = 5, direct = TRUE) {
  # Input validation
  if (!is.numeric(x) || !is.numeric(y)) {
    stop("Both 'x' and 'y' must be numeric vectors.")
  }
  if (length(x) != length(y)) {
    stop("'x' and 'y' must be of the same length.")
  }
  if (!is.numeric(k) || k <= 0 || k != round(k)) {
    stop("'k' must be a positive integer.")
  }
  if (!is.logical(direct)) {
    stop("'direct' must be a logical value.")
  }

  # Compute mutual information
  FNN::mutinfo(x, y, k = k, direct = direct)
}
